/////////////////////////////////////////////////////////////////////////////
//
//	File: QzTest.cpp
//
//	$Header: /Projects/QzTest/QzTest.cpp  3  2009/9/14 5:45:53p  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzInterface.h"
#include "QzRenderer.h"
#include "QzTest.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	InstanceQzClient()
//
//	This function is used by the QzManager class to instance the app-specific
//	client object, which is responsible for drawing the contents of the
//	window.
//
QzClient* InstanceQzClient(void)
{
	return new QzTest;
}


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzTest::QzTest(void)
	:	m_MouseLeftDown(false),
		m_CtrlDown(false),
		m_Paused(true),
		m_MouseLocked(false),
		m_LockedX(100),
		m_LockedY(100),
		m_TimeStamp(0)
{
	// A worker thread is created to demonstrate how to create a worker
	// thread that uses the platform-independent QzThreadControl class.
	m_Thread.CreateWorkerThread();
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzTest::~QzTest(void)
{
	m_Thread.DestroyWorkerThread();
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetAppTitle()
//
//	This function returns the string that will be displayed on the window's
//	title bar.
//
void QzTest::GetAppTitle(Utf08_t buffer[], U32 bufferSize)
{
	UtfComposeASCII(buffer, bufferSize, "Qz Test App");
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetMaxFrameDelay()
//
//	Reports the maximum number of milliseconds that should ellapse between
//	redrawing the window if the window is not dirty/invalidated.  When the
//	window is marked as dirty, it will be redrawn as soon as the app gets
//	chance to redraw.
//
U32 QzTest::GetMaxFrameDelay(void)
{
	// Only force the window to redraw itself at most once per second.
	// This does not prevent the window from redrawing more often if the
	// SetDirtyFlag() method is called.
	return m_MouseLocked ? 10 : 1000;
}


/////////////////////////////////////////////////////////////////////////////
//
//	KeyPress()
//
void QzTest::KeyPress(U32 keyFlags, U32 key)
{
	switch (key) {
		case QzKey_Escape:
			g_pQzInterface->RequestAppTermination();
			break;

		case QzKey_1:
			if (m_MouseLocked) {
				m_MouseLocked = false;
				g_pQzInterface->SetMouseExclusive(false);
			}
			else {
				m_MouseLocked = true;
				g_pQzInterface->SetMouseExclusive(true);
			}
			break;

		case QzKey_Space:
			m_Paused = !m_Paused;
			break;

		case QzKey_A:
			QzAssertAlways("How are you gentlemen!!");
			break;

		case QzKey_R:
			m_TimeStamp = 0;
			break;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	KeyChar()
//
void QzTest::KeyChar(Utf32_t keyChar)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseMove()
//
//	Normal mouse position updates are received when the mouse is unlocked.
//
void QzTest::MouseMove(S32 x, S32 y)
{
	m_MouseX = x;
	m_MouseY = y;

	// Demostrate that the frame gets redrawn more often when clicking and
	// dragging with the left mouse button.
	if (m_MouseLeftDown || m_MouseLocked) {
		g_pQzInterface->SetDirtyFlag();
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseDelta()
//
//	Delta values are only received when the mouse pointer is locked.
//
void QzTest::MouseDelta(S32 dx, S32 dy)
{
	m_LockedX += dx;
	m_LockedY += dy;

	// Demonstrate that the frame gets redrawn more often when moving the
	// mouse while the mouse is locked to the window (e.g., the kind of
	// logic used in an FPS game).
	if (m_MouseLocked) {
		g_pQzInterface->SetDirtyFlag();
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseClick()
//
void QzTest::MouseClick(QzMouseButton_t button, QzMouseClick_t click, U32 flags, S32 x, S32 y)
{
	if (QzMouseButton_Left == button) {
		if (QzMouseClick_Down == click) {
			m_MouseLeftDown = true;
		}
		else {
			m_MouseLeftDown = false;
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	MouseWheel()
//
void QzTest::MouseWheel(S32 delta)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	Render()
//
void QzTest::Render(QzRenderer *pRenderer)
{
//	S32 width  = pRenderer->GetWidth();
//	S32 height = pRenderer->GetHeight();

	pRenderer->ClearFrameBuffer();

	// Prepare to render using parallel (orthographic) projection, with the
	// coordinate scaled 1:1 with pixel coordinates.  The upper-level corner
	// is set up as coordinate (0,0).
	pRenderer->PrepareParallel();

	// When the mouse is locked, the mouse pointer is hidden.  Display a
	// small quad on the screen to indicate the position of the mouse.
	if (m_MouseLocked) {
		QzVector_t poly[4];
		poly[0].X = float(m_LockedX - 2);
		poly[1].X = float(m_LockedX + 2);
		poly[2].X = float(m_LockedX + 2);
		poly[3].X = float(m_LockedX - 2);
		poly[0].Y = float(m_LockedY - 2);
		poly[1].Y = float(m_LockedY - 2);
		poly[2].Y = float(m_LockedY + 2);
		poly[3].Y = float(m_LockedY + 2);
		poly[0].Z = 0.0f;
		poly[1].Z = 0.0f;
		poly[2].Z = 0.0f;
		poly[3].Z = 0.0f;
		pRenderer->DrawPolygon(poly, 4);
	}


	// Build up a string that will be displayed in the upper-left corner of
	// the screen.  This contains a timer, a rotating symbol, and the word
	// "[Paused]" when the timer is not counting.

	Utf08_t msg[256];
	Utf08_t spin[2];
	spin[1] = '\0';

	UtfFormat fmt;
	fmt.AddInt(m_TimeStamp / 1000);
	fmt.AddInt(m_TimeStamp % 1000);
	fmt.Generate(msg, ArraySize(msg), reinterpret_cast<const Utf08_t*>("%1w3;.%2w3p0; "));

	// The current counter state of the worker thread is used to control
	// the orientation of a rotating character.
	switch (m_Thread.GetCounter()) {
		case 0:  spin[0] = '-'; break;
		case 1:  spin[0] = '\\'; break;
		case 2:  spin[0] = '|'; break;
		default: spin[0] = '/'; break;
	}

	UtfAppend(msg, ArraySize(msg), spin);

	if (m_Paused) {
		UtfAppend(msg, ArraySize(msg), reinterpret_cast<const Utf08_t*>(" [Paused]"));
	}
	else {
		m_TimeStamp += g_pQzInterface->GetFrameDuration();
		if (m_TimeStamp > 10000) {
			m_TimeStamp = 10000;
		}
	}

	pRenderer->FontRender(QzFont_MonospaceLo, 10, 10, msg);
}




