/////////////////////////////////////////////////////////////////////////////
//
//	File: QzLogUtil.cpp
//
//	$Header: /Projects/QzLogUtilMac/QzLogUtil.cpp  1  2009/9/9 9:42:25a  Lee $
//
//
/////////////////////////////////////////////////////////////////////////////


#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <unistd.h>


typedef unsigned char      U08;
typedef unsigned long      U32;


// This header is stored at the beginning of shared memory.  Since shared
// memory is used as a ring buffer, these fields indicate the position of
// the data within the ring buffer.
//
struct QzLogInfo_t
{
	U32 Enable;        // flag used to disable the writing of log messages
	U32 ErrorCount;    // how many error messages have been logged
	U32 MaxByteCount;  // total size of shared memory buffer
	U32 ByteCount;     // total number of bytes in the ring buffer
	U32 ByteOffset;    // offset to the start of the ring
};


/////////////////////////////////////////////////////////////////////////////
//
//	SaveLog()
//
//	Writes the contents of shared memory out to a text file.  The default
//	assumption is that the text file will contain UTF-8 data.  However, if
//	all of the text is 7-bit ASCII, this will create a plain text file.
//
void SaveLog(char filename[], char *pAddress)
{
	QzLogInfo_t *pInfo = reinterpret_cast<QzLogInfo_t*>(pAddress);

	FILE *pFile = fopen(filename, "w");

	if (pFile != NULL) {
		// Shared memory is treated as a ring buffer.  The header at
		// the start of shared memory tells us what range of data in
		// the buffer contains out logging information.
		U32 byteOffset = pInfo->ByteOffset;
		U32 byteCount  = pInfo->ByteCount;
		U32 maxCount   = pInfo->MaxByteCount;

		// Skip over the header at the start of shared memory to get
		// to the start of the actual log text.
		char *pStart = reinterpret_cast<char*>(pAddress + sizeof(QzLogInfo_t));

		// Make life easier by allocating a temp buffer, with extra
		// space for a '\0' at the end of the buffer.  This allows
		// us to ignore any logic problems that will arise if a
		// multi-byte UTF-8 symbol is split across the end of the
		// ring buffer.
		char *pTemp = new char[byteCount + 1];

		// If the log data never wrapped around the end of shared
		// memory, we can simply memcpy() the string.
		if (byteCount < maxCount) {
			memcpy(pTemp, pStart, byteCount);
		}

		// Otherwise the data wraps around the end of the ring buffer,
		// so two memcpy's are required to piece the whole block of
		// log data into a continuous buffer.
		else {
			U32 count1 = maxCount - byteOffset;
			U32 count2 = byteOffset;
			memcpy(pTemp, pStart + byteOffset, count1);
			memcpy(pTemp + count1, pStart, count2);
		}

		// Terminate the buffer.  In this case it is not strictly
		// necessary, doing so can avoid any craziness that may arise
		// from trying to view a non-terminated string in a debugger.
		pTemp[byteCount] = '\0';

		U32  skip   = 0;
		bool isUTF8 = false;

		// It's possible that the oldest byte in the buffer is from the
		// middle of a multi-byte sequence.  This will skip any middle
		// bytes, looking for the first 7-bit ASCII byte, or the first
		// byte of a multi-byte sequence.  This should not be more than
		// three bytes into the buffer.  Normally, it should be the very
		// first byte, since almost no multi-byte symbols are used in
		// logging.
		for (U32 i = 0; i < byteCount; ++i) {
			if (0x80 != (0xC0 & pTemp[i])) {
				skip = i;
				break;
			}

			if (0 != (0x80 & pTemp[i])) {
				isUTF8 = true;
			}
		}

		// This three-byte sequence denotes a UTF-8 file.  But we only
		// need to write it if there are any 8-bit symbols.  If all of
		// the text is 7-bit ASCII, we can skip the marker, since WordPad
		// has problems with reading UTF-8 files (it appears to read data
		// in 8KB chunks, and glitches one of the bytes near the end of
		// each chunk).
		//
		if (isUTF8) {
			U08 marker[3];
			marker[0] = 0xEF;
			marker[1] = 0xBB;
			marker[2] = 0xBF;

			fwrite(marker, 1, 3, pFile);
		}

		// Write all of the log to disk, skipping any partial UTF-8 symbol
		// from the start of the buffer.
		fwrite(pTemp + skip, 1, byteCount - skip, pFile);

		fclose(pFile);

		printf("Log Size: %d bytes\n", byteCount + 3 - skip);

		delete [] pTemp;
	}
	else {
		printf("Unable to open file \"%s\"\n", filename);
	}

	if (0 != pInfo->ErrorCount) {
		if (pInfo->ErrorCount > 1) {
			printf("There were %d errors recorded in the log.\n", pInfo->ErrorCount);
		}
		else {
			printf("There was 1 error recorded in the log.\n");
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	CreateAndWait()
//
//	Creates a shared memory buffer, then sits in a loop.  The user can type
//	in "save" from the console window to write the contents of shared memory
//	out to disk, or "quit" to terminate the app.
//
void CreateAndWait(char shareName[], U32 byteCount, char logname[])
{
	// Call shm_open() to create a virtual file that will be used to back
	// the shared memory.  This file is created without any physical
	// backing, since we don't want to persist the data to disk, nor do we
	// want to run the performance risk of writing to a physical device
	// when logging large amounts of data.
	//
	// Note that for cross-platform compatibility, the return result is
	// being stored in a void* pointer instead of an int.
	//
	int hShare = shm_open(shareName, O_RDWR | O_CREAT, S_IRWXU);

	// Failure returns -1 instead of NULL.
	if (-1 == hShare) {
		printf("Error: could not create shared object\n");
		shm_unlink(shareName);
		return;
	}

	// This call will set the size of the shared buffer.
	ftruncate(hShare, byteCount);

	// Next we need to map the system memory into the virtual address
	// space of this process.  Again we need to specify read and
	// write permission to avoid page faults when writing to the
	// address space.
	//
	char *pAddress = (char*)mmap(NULL, byteCount, PROT_READ | PROT_WRITE, MAP_SHARED, hShare, 0);

	if (NULL == pAddress) {
		close(hShare);
		shm_unlink(shareName);
		return;
	}

	// Initialize the header at the beginning of shared memory.
	QzLogInfo_t *pInfo  = reinterpret_cast<QzLogInfo_t*>(pAddress);
	pInfo->Enable       = 1;
	pInfo->ErrorCount   = 0;
	pInfo->MaxByteCount = byteCount - sizeof(QzLogInfo_t);
	pInfo->ByteCount    = 0;
	pInfo->ByteOffset   = 0;


	//
	// Now sit in a loop waiting for console input.
	//

	char cmdLine[256];

	for (;;) {
		cmdLine[0] = '\0';
		fgets(cmdLine, 256, stdin);

		for (U32 i = 0; i < 256; ++i) {
			if ('\n' == cmdLine[i]) {
				cmdLine[i] = '\0';
				break;
			}
		}

		if (0 == strcmp("quit", cmdLine)) {
			break;
		}

		if (0 == strcmp("save", cmdLine)) {
			SaveLog(logname, pAddress);
		}
	}

	SaveLog(logname, pAddress);

	munmap(pAddress, byteCount);
	close(hShare);

	// This call will force the shared memory to be deleted from the system,
	// otherwise the memory file will linger around until the system is
	// rebooted.
	shm_unlink(shareName);
}


/////////////////////////////////////////////////////////////////////////////
//
//	main()
//
int main(int argc, char *argv[])
{
	char dirname[1024];

	char *logname = "trace.txt";

	printf("Logger Enabled\n");

	getcwd(dirname, 1024);

	// Display a string indicating where the file will be saved.
	printf("   cwd = \"%s/%s\"\n", dirname, logname);

	CreateAndWait("/QzLog", 4 * 1024 * 1024, logname);

	printf("Logger Terminated\n");

    return 0;
}




