/////////////////////////////////////////////////////////////////////////////
//
//	File: AudioTest.cpp
//
//	$Header: /Projects/QzAudioTest/AudioTest.cpp  1  2009/9/9 9:41:13a  Lee $
//
//
/////////////////////////////////////////////////////////////////////////////


#include <signal.h>
#include "QzCommon.h"
#include "QzLogger.h"
#include "QzOggDecoder.h"
#include "QzSoundDriver.h"
#include "UtfData.h"


// This is the maximum queue depth.  The actual look-ahead value is based
// on the sample rate; this value serves as the hard limit since this is
// the size of the scratch buffer used for mixing.
#define c_MaxQueueDepth		24000


/////////////////////////////////////////////////////////////////////////////
//
//	QzAssertHandler()
//
//	Custom assertion handler.  Deals with assertions without throwing up a
//	dialog, which doesn't necessary work right when running full-screen.
//
void QzAssertHandler(char message[], U32 lineNum, char file[])
{
	char *pattern = "Assert: line %1;, file %2;\n\n%3;\n";

	UtfFormat fmt;
	fmt.AddInt(lineNum);
	fmt.AddString(file);
	fmt.AddString(message);

	// Log the message.
	LogMessage(pattern, fmt);

	// Close the log file so we don't lose the assertion message.
	SafeDelete(g_pLog);

	// Use DebugBreak to launch the debugger ...
//	DebugBreak();

	// ... or just kill the app and drop back to the desktop.
	raise(SIGABRT);
}


U32  g_SampleRate = 44100;
U32  g_ToneFreq   =  1000;
U32  g_SineOffset =     0;
U32  g_PanPhase   = 44100 * 3;
U32  g_PanOffset  =     0;


/////////////////////////////////////////////////////////////////////////////
//
//	GenerateSamples()
//
//	This function will be used to generate audio data when we do not have an
//	OGG file to play.  It creates a sine wave at a fixed frequency (default
//	is 1000 Hz), which pans between left and right so there is some noticable
//	variation over time.
//
void GenerateSamples(S16 *pSamples, U32 sliceCount)
{
	float left, right;
	float separation = 0.875f;

	for (U32 sliceNum = 0; sliceNum < sliceCount; ++sliceNum) {
		// Compute a constant tone (default 1000 Hz tone), since a pure
		// sine wave does a very good job of revealing any problems with
		// discontinuities in the audio data.  The noise in a recorded
		// sound may mask some of the problems if audio data is not being
		// buffered correctly.  However, a pure sine wave is more likely
		// to make any glitches detectable as pops.
		//
		float wave = sinf(c_PI * 2.0f * float(g_ToneFreq) * float(g_SineOffset) / float(g_SampleRate));
		g_SineOffset = (g_SineOffset + 1) % g_SampleRate;

		// Compute a left/right pan effect so we have some variation on
		// the two channels over time.
		//
		float pan = sinf(c_PI * 2.0f * float(g_PanOffset) / float(g_PanPhase));
		g_PanOffset = (g_PanOffset + 1) % g_PanPhase;

		left  = 1.0f;
		right = 1.0f;

		if (pan >= -0.5f) {
			left -= (0.5f + pan) * (separation / 1.5f);
		}
		if (pan <= 0.5f) {
			right -= (0.5f - pan) * (separation / 1.5f);
		}

		pSamples[sliceNum*2  ] = S16(QzFloatToInt(10000.0f * wave * left));
		pSamples[sliceNum*2+1] = S16(QzFloatToInt(10000.0f * wave * right));
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	main()
//
int main(void)
{
	QzSystemInit();
	UtfInitialize();

	QzLogger *pLog = new QzLogger("QzLog");
	g_pLog = pLog;

	// Only enable file logging if the memory logging option is not available.
	// This typically means the logger utility has not been started.
	if (!g_pLog->MemoryLoggingAvailable()) {
		g_pLog->Open(reinterpret_cast<const Utf08_t*>("trace.txt"));
	}

	// Try to load the sample OGG file that will be played back.  This file
	// is found in the project directory, which works fine under DevStudio.
	// However, under Xcode, the default working directory will have to be
	// changed to a custom setting by Executables -> QzAudioTest -> Get Info.
	// When running the .app by itself, it will never see the OGG file.
	U32  oggSize = 0;
	U08* pOgg    = QzReadFileToBuffer(reinterpret_cast<const Utf08_t*>("morse.ogg"), oggSize);

	QzOggDecoder decoder;

	// Default to the standard CD audio sampling rate.
	// We'll override this if an OGG file is loaded for playback.
	U32 sampleRate = 44100;

	// If the OGG file could not be opened, or does not contain stereo data,
	// discard the file and fall back to generating a pure sine wave.
	if (NULL != pOgg) {
		if (decoder.Start(pOgg, oggSize)) {
			if (2 == decoder.ChannelCount()) {
				sampleRate = decoder.SampleRate();

				// For completeness, update these values, even though we are not
				// going to be using them if we have an OGG file to play back.
				g_SampleRate = sampleRate;
				g_PanPhase   = sampleRate * 3;
			}
			else {
				SafeDeleteArray(pOgg);
			}
		}
		else {
			SafeDeleteArray(pOgg);
		}
	}

	// Create a buffer where the audio data will be assembled before it
	// gets written to output.
	S16 *pScratch = new S16[c_MaxQueueDepth * 2];

	QzSoundDriver driver;

	// Try to keep a minimum amount of audio data queued up at all times.
	// This really only matters for DirectSound, since we have full one-second
	// buffer, but generally only need to keep 75-100 milliseconds of audio
	// queued up in it.  This is irrelevant for AudioQueue on Mac, which
	// uses a limited number of fixed-sized buffers, so look-ahead never
	// can go beyond the duration of data stored in those buffers.
	//
	// The nice thing about DirectSound is that we can write the audio data
	// directly into the output buffer.  This allows low-latency responsiveness
	// for audio output, something that is very useful for games or any other
	// app that wants a sound to be mixed into the output buffer without any
	// obvious lag.  If you take the alternate approach of allocating a separate
	// buffer for each sound being played, this is less of an issue, but there
	// typically are limits on the maximum number of buffers that can be played
	// simultaneously.
	//
	// The look-ahead duration is in milliseconds.
	//
	U32 lookaheadDuration = 100;

	// Compute the number of slices we need to keep buffered to maintain the
	// requested number of milliseconds of audio data.
	U32 lookahead = (lookaheadDuration * sampleRate) / 1000;

	// Limit the amount of data to avoid overflowing pScratch[].
	if (lookahead > c_MaxQueueDepth) {
		lookahead = c_MaxQueueDepth;
	}

	// For Mac's AudioQueue, we're using fixed-size buffers.  No matter what
	// the lookahead value is, we must always write this amount of data,
	// otherwise we won't be able to fill a whole buffer per call to
	// WriteToBuffer().
	//
	// For DirectSound, this value is irrelevant, since we can safely write
	// as little or as much audio as we want per call to WriteToBuffer().
	//
	if (lookahead < driver.MinWriteCount()) {
		lookahead = driver.MinWriteCount();
	}

	// Initialize the driver.  This allocates all of the buffers and sets up to
	// play at the specified sampling rate.
	driver.Init(sampleRate);

	// Start playing audio.  This begins consuming the pre-queued about of
	// silence sitting in the output buffers.
	//
	// After we make this call, we must continually feed more audio data to
	// the driver to avoid playback problems.  If Mac's AudioQueue starves,
	// audio will drop out.  But DirectSound will keep playing old data from
	// its output buffer, causing glitching or stuttering effects (sometimes
	// described as "robot noise").
	//
	driver.Start();

	U32 t0 = QzGetMilliseconds();
	U32 t1 = t0;

	U32 update = t0;

	// For demo purposes, play 10 seconds worth of sound.
	while ((t1 - t0) < 10000) {

		// How many empty buffers are there to fill up?  Make this
		// call only once, and do it before we enter the loop.
		// If we tried to repeatedly call this inside the inner
		// loop, we would end up in an infinite loop when running
		// with the DirectSound version of the driver.
		U32 freeCount = driver.FreeBufferCount();

		for (U32 i = 0; i < freeCount; ++i) {

			// Find out how much data is currently stored in the output buffer.
			U32 sliceCount = driver.UpdatePosition(lookahead);

			// If the amount of audio data is less than the desired
			// lookahead amount (and it always should be), we'll need
			// to mix up more audio to fill in the empty space in the
			// buffer.
			if (sliceCount > 0) {
				// If we don't have an OGG file to play, generate some sine
				// wave data into the buffer instead.
				if (NULL == pOgg) {
					GenerateSamples(pScratch, sliceCount);
				}
				else {
					U32 readCount = decoder.Read(pScratch, sliceCount);

					// If we did not get the full amount of data requested,
					// then we've reached the end of the file.  Loop back
					// to the start of the file and grab some more.
					//
					// Note that this breaks if the duration of the OGG file
					// is smaller than the buffer we're trying to fill.  We
					// would need to loop for that, but sound clips that
					// small probably don't need to be compressed (or at the
					// very least should be uncompressed once when loaded,
					// and played back from the uncompressed buffer not the
					// encoded OGG data).  Real-time decoding of OGG should
					// be reserved for large files, like music or voice.
					//
					if (readCount < sliceCount) {
						decoder.Loop();
						decoder.Read(pScratch + readCount, sliceCount - readCount);
					}
				}

				driver.WriteToBuffer(pScratch, sliceCount);
			}
		}

		// Wake up about 30 times a second to feed in more audio data.
		// At a 44,100 Hz sample rate, sleeping for 30 milliseconds
		// will consume about 1,300 slices per iteration of the loop.
		QzSleep(30);

		t1 = QzGetMilliseconds();

		// Every two seconds, display the current CPU usage.
		if ((t1 - update) >= 2000) {
			float cpuUsage = QzGetProcessorUsage();
			printf("CPU: %1.3f%%\n", 100.0f * cpuUsage);
			update += 2000;
		}
	}

	driver.Stop();

	if (NULL != pOgg) {
		decoder.Stop();
		SafeDeleteArray(pOgg);
	}

	SafeDeleteArray(pScratch);

	g_pLog = NULL;

	SafeDelete(pLog);

	UtfUninitialize();
	QzSystemUninit();

	printf("Done\n");

	return 0;
}


