/////////////////////////////////////////////////////////////////////////////
//
//	File: QzVector.cpp
//
//	$Header: /TS/TsGui/QzVector.cpp  33  2009/9/8 4:15:58p  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzVector.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	Normalize()
//
void QzVector::Normalize(void)
{
	float lengthSquared = (m_X * m_X) + (m_Y * m_Y) + (m_Z * m_Z);

	// Don't normalize the vector if it's close to being normalized already.
	//
	// Note that we're using QzEqualFloats20() here, since it's comparing the
	// square of the length.  Values close to 1.0 will be further from 1.0
	// before squaring, so we don't need as much precision on the comparison.
	//
	// Note: This has been verified by the unit test TestVector(), which
	// shows that QzEqualFloats20() will reliably detect the lengthSquared is
	// 1.0f when dealing with a vector that was already normalized once.
	//
	if (false == QzEqualFloats20(1.0f, lengthSquared)) {
		float length = sqrtf(lengthSquared);

		if (false == QzFloatIsZero20(length)) {
			float norm = 1.0f / length;
			m_X *= norm;
			m_Y *= norm;
			m_Z *= norm;
		}
		else {
			m_X = 1.0f;
			m_Y = 0.0f;
			m_Z = 0.0f;
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Average()
//
void QzVector::Average(const QzVector &v1, const QzVector &v2)
{
	m_X = (v1.m_X + v2.m_X) * 0.5f;
	m_Y = (v1.m_Y + v2.m_Y) * 0.5f;
	m_Z = (v1.m_Z + v2.m_Z) * 0.5f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DotProduct()
//
float QzVector::DotProduct(const QzVector &v) const
{
	return ((m_X * v.m_X) + (m_Y * v.m_Y) + (m_Z * v.m_Z));
}


/////////////////////////////////////////////////////////////////////////////
//
//	CrossProduct()
//
void QzVector::CrossProduct(const QzVector_t &v1, const QzVector_t &v2, const QzVector_t &v3)
{
	QzVector vector1, vector2;

	vector1.m_X = v2.X - v1.X;
	vector1.m_Y = v2.Y - v1.Y;
	vector1.m_Z = v2.Z - v1.Z;

	vector2.m_X = v3.X - v2.X;
	vector2.m_Y = v3.Y - v2.Y;
	vector2.m_Z = v3.Z - v2.Z;

    m_X = vector1.m_Y * vector2.m_Z - vector1.m_Z * vector2.m_Y;
    m_Y = vector1.m_Z * vector2.m_X - vector1.m_X * vector2.m_Z;
    m_Z = vector1.m_X * vector2.m_Y - vector1.m_Y * vector2.m_X;
}


/////////////////////////////////////////////////////////////////////////////
//
//	CrossProduct()
//
void QzVector::CrossProduct(const QzVector &v1, const QzVector &v2, const QzVector &v3)
{
	QzVector vector1 = v2 - v1;
	QzVector vector2 = v3 - v2;

	CrossProduct(vector1, vector2);
}


/////////////////////////////////////////////////////////////////////////////
//
//	CrossProduct()
//
void QzVector::CrossProduct(const QzVector &v1, const QzVector &v2)
{
    m_X = v1.m_Y * v2.m_Z - v1.m_Z * v2.m_Y;
    m_Y = v1.m_Z * v2.m_X - v1.m_X * v2.m_Z;
    m_Z = v1.m_X * v2.m_Y - v1.m_Y * v2.m_X;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DotProductSegmentByDirection()
//
//	Returns the dot product between the current vertex and a segment defined
//	by a starting point s0 and the segment's direction normal.
//
float QzVector::DotProductSegmentByDirection(const QzVector &s0, const QzVector &direction) const
{
	QzVector temp = (*this - s0);

	return temp.DotProduct(direction);
}


/////////////////////////////////////////////////////////////////////////////
//
//	DotProductSegmentByPoints()
//
//	Returns the dot product between the current vertex and a segment defined
//	by two points, s0 and s1, and assumes (for where it matters) that the
//	segment moves from s0 to s1.
//
float QzVector::DotProductSegmentByPoints(const QzVector &s0, const QzVector &s1) const
{
	QzVector temp = (*this - s0);

	return temp.DotProduct(s1 - s0);
}


/////////////////////////////////////////////////////////////////////////////
//
//	DistanceToSegment()
//
//	Returns the distance between the current vertex and a segment defined by
//	two points, s0 and s1, and assumes (for where it matters) that the segment
//	moves from s0 to s1.
//
float QzVector::DistanceToSegment(const QzVector &s0, const QzVector &s1) const
{
	QzVector nearest;
	nearest.NearestPointOnSegment(s0, s1, *this);

	return Distance(nearest);
}


/////////////////////////////////////////////////////////////////////////////
//
//	NearestPointOnLine()
//
//	Projects the given point to the nearest location on the line that
//	intersects the points s0 and s1.
//
void QzVector::NearestPointOnLine(const QzVector &s0, const QzVector &s1, const QzVector &point)
{
	// Find the normalized direction vector of the line.
	QzVector normal = s1 - s0;
	normal.Normalize();

	// Compute the relative offset of the point from the line origin.
	QzVector offset = point - s0;

	// Parametric t indicates distance to nearest point along the direction vector.
	float t = offset.DotProduct(normal);

	*this = s0 + (normal * t);
}


/////////////////////////////////////////////////////////////////////////////
//
//	NearestPointOnSegment()
//
//	Projects the given point to the nearest location on the line segment
//	delimited by s0 and s1.
//
//	Returns false if the projected point is not between s0 and s1.
//
bool QzVector::NearestPointOnSegment(const QzVector &s0, const QzVector &s1, const QzVector &point)
{
	// Compute angle from the s0 end of the segment.
	float dot0 = point.DotProductSegmentByPoints(s0, s1);

	// If the distance is negative, then the nearest point is past this end of
	// the segment, so return false and set the nearest point to s0.
	if (dot0 <= 0.0f) {
		*this = s0;
		return false;
	}

	// Compute the angle from the s1 end of the segment.
	float dot1 = point.DotProductSegmentByPoints(s1, s0);

	// If this distance is negative, the nearest point is past the s1 end of the
	// segment, so set the nearest point to s1 and return
	if (dot1 <= 0.0f) {
		*this = s1;
		return false;
	}

	float interp = dot0 / (dot0 + dot1);

	// Interpolate the nearest point on the line using the ratio between the
	// two dot products.
	*this = s0 + ((s1 - s0) * interp);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	NearestPointOnPolygon()
//
bool QzVector::NearestPointOnPolygon(const QzVector verts[], const U32 vertCount, const QzVector point)
{
	// Normal vector for the polygon.
	QzVector normal;
	normal.CrossProduct(verts[0], verts[1], verts[2]);
	normal.Normalize();

	// Compute nearest point on the plane.
	QzVector displace = point - verts[0];
	float     distance = normal.DotProduct(displace);
	QzVector nearest  = point - (normal * distance);

	float dot = normal.DotProduct(nearest - QzVector(verts[0]));

	// Point is not on the plane.
	if (false == QzFloatIsZero10(dot)) {
		nearest = point;
	}

	for (U32 i = 0; i < vertCount; ++i) {
		QzVector cross;
		QzVector side1 = verts[(i + 1) % vertCount] - verts[i];
		QzVector side2 = nearest - verts[i];
		cross.CrossProduct(normal, side1);

		// If the point is on the wrong side of this edge, project the point
		// onto this edge.
		// NOTE: This is not good enough.  For a polygon with obtuse angles,
		// this may end up picking the wrong corner.  It will work fine for
		// quads.  Should test against the prev/next edge to see if it will
		// produce a different result.
		if (cross.DotProduct(side2) < 0.0f) {
			NearestPointOnSegment(verts[i], verts[(i + 1) % vertCount], nearest);
			return false;
		}
	}

	// If we fall out of the loop, then the computed nearest point is within
	// the polygon.
	*this = nearest;

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IsPointInPolygon()
//
//	Assumes that the given points are coplanar.  If this point is on that
//	plane, and within the convex area circumscribed by the vertices, this
//	will return true.
//
//	This also assumes that none of the vertices form collinear edges, which
//	will break the normal computation.  As long as the first three vertices
//	at not collinear, this will work.
//
bool QzVector::IsPointInPolygon(const QzVector verts[], const U32 vertCount)
{
	QzVector normal;
	normal.CrossProduct(verts[0], verts[1], verts[2]);
	normal.Normalize();

	float dot = normal.DotProduct(*this - QzVector(verts[0]));

	// Point is not on the plane.
	if (false == QzFloatIsZero10(dot)) {
		return false;
	}

	for (U32 i = 0; i < vertCount; ++i) {
		QzVector cross;
		QzVector side1 = verts[(i + 1) % vertCount] - verts[i];
		QzVector side2 = *this - verts[i];
		cross.CrossProduct(normal, side1);

		if (cross.DotProduct(side2) < 0.0f) {
			return false;
		}
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IntersectRaySphere()
//
//	Checks to see if the ray intersects with a sphere.  If so, it returns
//	true and sets this vector equal to the point of intersection.
//
//	start  - starting point of the ray
//	dir    - normalized direction of the ray
//	sphere - center of the sphere
//	radius - radius of the sphere
//
bool QzVector::IntersectRaySphere(const QzVector &start, const QzVector &dir, const QzVector &sphere, const float radius)
{
	QzVector offset = sphere - start;

	// Computes distance along the ray to point nearest the center of the sphere.
	float distance = dir.DotProduct(offset);

	// Nearest point on the ray is behind the starting point, so ignore it.
	if (distance <= 0.0f) {
		return false;
	}

	float offsetSquared = offset.DotProduct(offset);
	float radiusSquared = radius * radius;

	// The start of the ray is within the sphere.
	if (offsetSquared < radiusSquared) {
		*this = start;
		return true;
	}

	float hitSquared = radiusSquared - (offsetSquared - (distance * distance));

	// Ray does not intersect sphere.
	if (hitSquared < 0.0f) {
		return false;
	}

	// Distance along ray to first intersection with sphere.
	float hitDistance = distance - sqrtf(hitSquared);

	*this = start + (dir * hitDistance);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IntersectRayPolygon()
//
//	Takes a polygon delimited by <vertCount> vertices in <vert>, and attempts
//	to determine whether the given ray intersects the plane.  This can fail
//	if the ray is parallel to the plane.
//
//	Otherwise, it will set <this> equal to the point of intersection, then
//	test whether this points falls within the bounds of the polygon.  Due
//	to floating point precision, the accuracy is questionable if the point
//	of intersection is right at the edge of the polygon.
//
//	Returns false if the point is not in the polygon.
//
bool QzVector::IntersectRayPolygon(const QzVector &start, const QzVector &dir, const QzVector verts[], const U32 vertCount)
{
	QzVector normal;
	normal.CrossProduct(verts[0], verts[1], verts[2]);
	normal.Normalize();

	QzVector offset = start - QzVector(verts[0]);

	float bottom = normal.DotProduct(dir);

	// Ray is either parallel to the plane, or away from it.
	if (QzFloatIsZero20(bottom)) {
		*this = start;
		return false;
	}

	float distance = -normal.DotProduct(offset) / bottom;

	*this = start + (dir * distance);

	for (U32 i = 0; i < vertCount; ++i) {
		QzVector side1 = verts[(i + 1) % vertCount] - verts[i];
		QzVector side2 = *this - verts[i];

		QzVector cross;

		cross.CrossProduct(dir, side1);
		if (cross.DotProduct(side2) >= 0.0f)
			return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IntersectRayPlane()
//
//	Given three points in <verts> that define a plane, it projects the ray
//	onto the plane and computes the point at which the ray intersects the
//	plane.
//
//	Expects that <dir> is a normalized vector.
//
//	Returns false if the ray is parallel to the plane.
//
bool QzVector::IntersectRayPlane(const QzVector &start, const QzVector &dir, const QzVector verts[])
{
	QzVector normal;
	normal.CrossProduct(verts[0], verts[1], verts[2]);
	normal.Normalize();

	float bottom = normal.DotProduct(dir);

	// Segment is parallel to the plane.
	if (QzFloatIsZero20(bottom)) {
		return false;
	}

	QzVector offset   = start - verts[0];
	float    distance = -normal.DotProduct(offset) / bottom;

	*this = start + (dir * distance);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IntersectRayPlane()
//
//	Given three points in <verts> that define a plane, it projects the ray
//	onto the plane and computes the point at which the ray intersects the
//	plane.
//
//	Expects that <dir> is a normalized vector.
//
//	Returns false if the ray is parallel to the plane.
//
bool QzVector::IntersectRayPlane(const QzVector &start, const QzVector &dir, const QzVector &normal, const QzVector &surface)
{
	float bottom = normal.DotProduct(dir);

	// Segment is parallel to the plane.
	if (QzFloatIsZero20(bottom)) {
		return false;
	}

	QzVector offset = start - surface;

	float distance = -normal.DotProduct(offset) / bottom;

	*this = start + (dir * distance);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IntersectSegmentPolygon()
//
//	Given a segment and a planar polygon defined by <vertCount> points in <verts>,
//	computes the point of intersection of the segment and the polygon.  Tests
//	whether the segment crosses the plane between the <start> and <stop>
//	points of the segment, and whether the point of intersection is within
//	the bounds of the polygon.
//
//	Returns true only if the segment intersects the polygon.
//
bool QzVector::IntersectSegmentPolygon(const QzVector &start, const QzVector &stop, const QzVector verts[], const U32 vertCount)
{
	QzVector dir = stop - start;
	float length  = dir.Length();
	dir.Normalize();

	QzVector normal;
	normal.CrossProduct(verts[0], verts[1], verts[2]);
	normal.Normalize();

	QzVector offset = start - QzVector(verts[0]);

	float bottom = normal.DotProduct(dir);

	// Segment is either parallel to the plane, or away from it.
	if (QzFloatIsZero20(bottom)) {
		return false;
	}

	float distance = -normal.DotProduct(offset) / bottom;

	// If the distance to the plane off either end of the segment,
	// there is no intersection.
	if ((distance < 0.0f) || (distance > length)) {
		return false;
	}

	*this = start + (dir * distance);

	for (U32 i = 0; i < vertCount; ++i) {
		QzVector side1 = verts[(i + 1) % vertCount] - verts[i];
		QzVector side2 = *this - verts[i];

		QzVector cross;

		cross.CrossProduct(dir, side1);
		if (cross.DotProduct(side2) >= 0.0f) {
			return false;
		}
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IntersectSegmentPlane()
//
//	Takes a plane delimited by the three points in <verts> and tests whether
//	the given segment crosses the plane.  Returns true if the point of
//	intersection is between the <start> and <stop> points of the segment.
//
bool QzVector::IntersectSegmentPlane(const QzVector &start, const QzVector &stop, const QzVector verts[])
{
	QzVector dir = stop - start;
	float length  = dir.Length();
	dir.Normalize();

	QzVector normal;
	normal.CrossProduct(verts[0], verts[1], verts[2]);
	normal.Normalize();

	QzVector offset = start - verts[0];

	float bottom = normal.DotProduct(dir);

	// Segment is parallel to the plane.
	if (QzFloatIsZero20(bottom)) {
		return false;
	}

	// Distance from the starting point to the point of intersection on the plane.
	float distance = -normal.DotProduct(offset) / bottom;

	// If the distance to the plane off either end of the segment,
	// there is no intersection.
	if ((distance < 0.0f) || (distance > length)) {
		return false;
	}

	*this = start + (dir * distance);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Interpolate()
//
void QzVector::Interpolate(const QzVector &s0, const QzVector &s1, const float interp)
{
	*this = s0 + ((s1 - s0) * interp);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Equal()
//
bool QzVector::Equal(const QzVector &v) const
{
	return (QzEqualFloats10(v.m_X, m_X)
		 && QzEqualFloats10(v.m_Y, m_Y)
		 && QzEqualFloats10(v.m_Z, m_Z));
}


/////////////////////////////////////////////////////////////////////////////
//
//	RotateX()
//
void QzVector::RotateX(float angle)
{
	float y = m_Y;
	float z = m_Z;

	m_Y = y * cosf(angle) - z * sinf(angle);
	m_Z = y * sinf(angle) + z * cosf(angle);
}


/////////////////////////////////////////////////////////////////////////////
//
//	RotateY()
//
void QzVector::RotateY(float angle)
{
	float x = m_X;
	float z = m_Z;

	m_X =  x * cosf(angle) + z * sinf(angle);
	m_Z = -x * sinf(angle) + z * cosf(angle);
}


/////////////////////////////////////////////////////////////////////////////
//
//	RotateZ()
//
void QzVector::RotateZ(float angle)
{
	float x = m_X;
	float y = m_Y;

	m_X = x * cosf(angle) - y * sinf(angle);
	m_Y = x * sinf(angle) + y * cosf(angle);
}


/////////////////////////////////////////////////////////////////////////////
//
//	RotateArbitraryAxis()
//
//	Rotates the point about an arbitrary axis in 3D space.  The given axis
//	needs to be a normalized value -- it's assumed the axis was normalized
//	before this routine was called, since the same axis is frequently used
//	for multiple rotations.
//
//	The rotation is performed about the origin.  Rotations about a specific
//	location in 3D space require that the point be translated so the rotation
//	is centered at the origin, then translated back to the correct center of
//	rotation.
//
void QzVector::RotateArbitraryAxis(QzVector axis, float angle)
{
	float cosTheta = cosf(angle);
	float sinTheta = sinf(angle);
	float oneMinus = 1.0f - cosTheta;

	float baseX = m_X;
	float baseY = m_Y;
	float baseZ = m_Z;

	m_X  = baseX * (oneMinus * axis.m_X * axis.m_X + cosTheta);
	m_X += baseY * (oneMinus * axis.m_X * axis.m_Y - axis.m_Z * sinTheta);
	m_X += baseZ * (oneMinus * axis.m_X * axis.m_Z + axis.m_Y * sinTheta);

	m_Y  = baseX * (oneMinus * axis.m_Y * axis.m_X + axis.m_Z * sinTheta);
	m_Y += baseY * (oneMinus * axis.m_Y * axis.m_Y + cosTheta);
	m_Y += baseZ * (oneMinus * axis.m_Y * axis.m_Z - axis.m_X * sinTheta);

	m_Z  = baseX * (oneMinus * axis.m_Z * axis.m_X - axis.m_Y * sinTheta);
	m_Z += baseY * (oneMinus * axis.m_Z * axis.m_Y + axis.m_X * sinTheta);
	m_Z += baseZ * (oneMinus * axis.m_Z * axis.m_Z + cosTheta);
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetIntVector()
//
QzIntVector_t QzVector::GetIntVector(void) const
{
	QzIntVector_t v;
	v.X = QzFloatToInt(m_X * float(QzFixedPoint_One));
	v.Y = QzFloatToInt(m_Y * float(QzFixedPoint_One));
	v.Z = QzFloatToInt(m_Z * float(QzFixedPoint_One));
	return v;
}
