/////////////////////////////////////////////////////////////////////////////
//
//	File: QzSoundDriverWin.cpp
//
//	$Header: /Projects/Qz/QzSoundDriverWin.cpp  1  2009/9/9 9:02:04a  Lee $
//
//
//	This implements the Windows version of a sound player, using DirectSound
//	to feed 16-bit PCM audio to the speakers.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzSoundDriver.h"
#include <dsound.h>


#pragma comment(lib, "dsound")


typedef IDirectSound8					DxSound_t;
typedef IDirectSoundBuffer8				DsBuffer_t;
typedef IDirectSound3DListener			DsListener_t;


// Define this here, so we don't need to link in yet another DX library.
static const GUID g_SoundBuffer8Guid =
{
	0x6825a449, 0x7524, 0x4d82, 0x92, 0x0f, 0x50, 0xe3, 0x6a, 0xb3, 0xab, 0x1e
};



// A window handle is required since sound playback will be attached to a
// specific window, since DirectSound likes to silence the output when its
// parent window loses mouse/keyboard input focus.
extern HWND g_hWindow;


struct DirectSoundContext_t
{
	DSCAPS		DeviceCaps;
	DxSound_t*	pDxSound;
	DsBuffer_t*	pBuffer;
	U32			SampleRate;
	U32			ChannelCount;
	U32			BufferSize;
	U32			Position;
	U32			WriteOffset;
	U32			BytesRemaining;
};


/////////////////////////////////////////////////////////////////////////////
//
struct DSErrorInfo_t
{
	HRESULT	Code;
	char*   Label;
};

// List of standard errors that can be returned from DirectSound.
// This allows for a more relevant description than otherwise returned
// from the FormatMessage() function.
//
DSErrorInfo_t g_DSErrorInfo[] =
{
	{ DS_OK,					"DS_OK: The method succeeded." },
	{ DS_NO_VIRTUALIZATION,		"DS_NO_VIRTUALIZATION: The buffer was created, but another 3-D algorithm was substituted." },
//	{ DS_INCOMPLETE,			"DS_INCOMPLETE: The method succeeded, but not all the optional effects were obtained." },
	{ DSERR_ACCESSDENIED,		"DSERR_ACCESSDENIED: The request failed because access was denied." },
	{ DSERR_ALLOCATED,			"DSERR_ALLOCATED: The request failed because resources, such as a priority level, were already in use by another caller." },
	{ DSERR_ALREADYINITIALIZED,	"DSERR_ALREADYINITIALIZED: The object is already initialized." },
	{ DSERR_BADFORMAT,			"DSERR_BADFORMAT: The specified wave format is not supported." },
	{ DSERR_BADSENDBUFFERGUID,	"DSERR_BADSENDBUFFERGUID: The GUID specified in an audiopath file does not match a valid mix-in buffer." },
	{ DSERR_BUFFERLOST,			"DSERR_BUFFERLOST: The buffer memory has been lost and must be restored." },
	{ DSERR_BUFFERTOOSMALL,		"DSERR_BUFFERTOOSMALL: The buffer size is not great enough to enable effects processing." },
	{ DSERR_CONTROLUNAVAIL,		"DSERR_CONTROLUNAVAIL: The buffer control (volume, pan, and so on) requested by the caller is not available. Controls must be specified when the buffer is created, using the dwFlags member of DSBUFFERDESC." },
	{ DSERR_DS8_REQUIRED,		"DSERR_DS8_REQUIRED: A DirectSound object of class CLSID_DirectSound8 or later is required for the requested functionality. For more information, see IDirectSound8 Interface." },
	{ DSERR_FXUNAVAILABLE,		"DSERR_FXUNAVAILABLE: The effects requested could not be found on the system, or they are in the wrong order or in the wrong location; for example, an effect expected in hardware was found in software." },
	{ DSERR_GENERIC,			"DSERR_GENERIC: An undetermined error occurred inside the DirectSound subsystem." },
	{ DSERR_INVALIDCALL,		"DSERR_INVALIDCALL: This function is not valid for the current state of this object." },
	{ DSERR_INVALIDPARAM,		"DSERR_INVALIDPARAM: An invalid parameter was passed to the returning function." },
	{ DSERR_NOAGGREGATION,		"DSERR_NOAGGREGATION: The object does not support aggregation." },
	{ DSERR_NODRIVER,			"DSERR_NODRIVER: No sound driver is available for use, or the given GUID is not a valid DirectSound device ID." },
	{ DSERR_NOINTERFACE,		"DSERR_NOINTERFACE: The requested COM interface is not available." },
	{ DSERR_OBJECTNOTFOUND,		"DSERR_OBJECTNOTFOUND: The requested object was not found." },
	{ DSERR_OTHERAPPHASPRIO,	"DSERR_OTHERAPPHASPRIO: Another application has a higher priority level, preventing this call from succeeding. " },
	{ DSERR_OUTOFMEMORY,		"DSERR_OUTOFMEMORY: The DirectSound subsystem could not allocate sufficient memory to complete the caller's request." },
	{ DSERR_PRIOLEVELNEEDED,	"DSERR_PRIOLEVELNEEDED: A cooperative level of DSSCL_PRIORITY or higher is required." },
	{ DSERR_SENDLOOP,			"DSERR_SENDLOOP: A circular loop of send effects was detected." },
	{ DSERR_UNINITIALIZED,		"DSERR_UNINITIALIZED: The IDirectSound8::Initialize method has not been called or has not been called successfully before other methods were called." },
	{ DSERR_UNSUPPORTED,		"DSERR_UNSUPPORTED: The function called is not supported at this time." }
};


/////////////////////////////////////////////////////////////////////////////
//
//	HRESULTtoString()
//
//	Scans through the contents of g_DSErrorInfo[] for the given error code.
//	If the HRESULT is not in the table, it is probably not a DirectSound
//	error code.
//
static Utf08_t* HRESULTtoString(const HRESULT hr)
{
	for (U32 i = 0; i < ArraySize(g_DSErrorInfo); ++i) {
		if (hr == g_DSErrorInfo[i].Code) {
			return reinterpret_cast<Utf08_t*>(g_DSErrorInfo[i].Label);
		}
	}

	return NULL;
}


/////////////////////////////////////////////////////////////////////////////
//
//	PrintHRESULT()
//
//	This takes an HRESULT that it attempts to convert into a meaningful error
//	message.  Unfortunately, the D3D error functions do nothing more than
//	provide the symbolic name of the HRESULT instead of a useful description.
//
//	So this will try to use a meaningful string from the g_DSErrorInfo[]
//	array, or fall back to FormatMessage() if the HRESULT is unknown.
//
static void PrintHRESULT(char message[], const HRESULT hr)
{
	Utf16_t wideBuffer[512];
	Utf08_t buffer[512];

	Utf08_t *pMessage = HRESULTtoString(hr);

	if (NULL != pMessage) {
		SafeStrCopy(buffer, pMessage);
	}
	else if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
			NULL,
			hr,
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
			wideBuffer,
			ArraySize(wideBuffer),
			NULL))
	{
		wideBuffer[ArraySize(wideBuffer)-1] = '\0';
		UtfCompose16to08(buffer, ArraySize(buffer), wideBuffer);
	}
	else {
		SafeStrCopy(buffer, reinterpret_cast<Utf08_t*>("unknown DirectSound error"));
	}

	UtfFormat fmt;
	fmt.AddString(message);
	fmt.AddInt(hr);
	fmt.AddString(buffer);
	LogErrorMessage("DirectSoundDriver::%1;, hr = 0x%2xw8p0; = %3;", fmt);
}


/////////////////////////////////////////////////////////////////////////////
//
//	DirectSoundEnum()
//
//	Callback handler used by DirectSoundEnumerate().  This function will be
//	called once for each DirectSound device in the system, which is generally
//	just two: the primary sound device and the physical device (such as a
//	SoundMAX or SoundBlaster).  These two devices are the same device, but
//	represented by different GUIDs.
//
//	We're not going to use this information for anything, but logging the
//	driver info can be useful for debugging purposes.
//
static BOOL CALLBACK DirectSoundEnum(GUID *pGUID, const wchar_t *pDesc, const wchar_t *pDriverName, void * /*pContext*/)
{
	Utf08_t desc08[256];
	Utf08_t name08[256];

	UtfCompose16to08(desc08, ArraySize(desc08), pDesc);
	UtfCompose16to08(name08, ArraySize(name08), pDriverName);

	UtfFormat fmt;
	fmt.AddString(desc08);
	LogMessage("DirectSound driver enum: %1;", fmt);

	if ((NULL != pDriverName) && ('\0' != pDriverName[0])) {
		UtfFormat fmt;
		fmt.AddString(name08);
		LogMessage("   driver = %1;", fmt);
	}

	if ((NULL != pDesc) && ('\0' != pDesc[0])) {
		UtfFormat fmt;
		fmt.AddString(desc08);
		LogMessage("   desc   = %1;", fmt);
	}

	if (NULL == pGUID) {
		LogMessage("   GUID   = NULL");
	}
	else {
		// Cannot handle 11 simultaneous parameters through UtfFormat.
		// So do this in two passes, generating the last part of the string
		// into a temp buffer.
		Utf08_t theRestOfIt[128];
		fmt.Reset();
		fmt.AddInt(pGUID->Data4[0]);
		fmt.AddInt(pGUID->Data4[1]);
		fmt.AddInt(pGUID->Data4[2]);
		fmt.AddInt(pGUID->Data4[3]);
		fmt.AddInt(pGUID->Data4[4]);
		fmt.AddInt(pGUID->Data4[5]);
		fmt.AddInt(pGUID->Data4[6]);
		fmt.AddInt(pGUID->Data4[7]);
		fmt.Generate(theRestOfIt, ArraySize(theRestOfIt), reinterpret_cast<Utf08_t*>("%1xw2p0;-%2xw2p0;-%3xw2p0;-%4xw2p0;-%5xw2p0;-%6xw2p0;-%7xw2p0;-%8xw2p0;"));

		fmt.Reset();
		fmt.AddInt(pGUID->Data1);
		fmt.AddInt(pGUID->Data2);
		fmt.AddInt(pGUID->Data3);
		fmt.AddString(theRestOfIt);
		LogMessage("   GUID   = %1xw8p0;-%2xw4p0;-%3xw4p0;-%4;", fmt);
	}

	return TRUE;
}


static void LogInt(char pattern[], S32 value)
{
	UtfFormat fmt;
	fmt.AddInt(value);
	LogMessage(pattern, fmt);
}


static void LogString(char pattern[], char value[])
{
	UtfFormat fmt;
	fmt.AddString(value);
	LogMessage(pattern, fmt);
}


#define LogCap(bits,flag) { LogString("    %1;: " #flag "", ((bits) & (flag)) ? "yes" : " no"); }



/////////////////////////////////////////////////////////////////////////////
//
//	CheckCaps()
//
void CheckCaps(DxSound_t *pDxSound, DSCAPS &deviceCaps, bool logCaps)
{
	memset(&deviceCaps, 0, sizeof(deviceCaps));
	deviceCaps.dwSize = sizeof(deviceCaps);

	HRESULT hr = pDxSound->GetCaps(&deviceCaps);
	if (FAILED(hr)) {
		PrintHRESULT("CheckCaps() could not retrieve caps", hr);
		return;
	}

	if (false == logCaps) {
		return;
	}

	LogMessage("Direct Sound Capabilities:");

	LogCap(deviceCaps.dwFlags, DSCAPS_CERTIFIED);
	LogCap(deviceCaps.dwFlags, DSCAPS_CONTINUOUSRATE);
	LogCap(deviceCaps.dwFlags, DSCAPS_EMULDRIVER);
	LogCap(deviceCaps.dwFlags, DSCAPS_PRIMARY16BIT);
	LogCap(deviceCaps.dwFlags, DSCAPS_PRIMARY8BIT);
	LogCap(deviceCaps.dwFlags, DSCAPS_PRIMARYMONO);
	LogCap(deviceCaps.dwFlags, DSCAPS_PRIMARYSTEREO);
	LogCap(deviceCaps.dwFlags, DSCAPS_SECONDARY16BIT);
	LogCap(deviceCaps.dwFlags, DSCAPS_SECONDARY8BIT);
	LogCap(deviceCaps.dwFlags, DSCAPS_SECONDARYMONO);
	LogCap(deviceCaps.dwFlags, DSCAPS_SECONDARYSTEREO);

	UtfFormat fmt;
	fmt.AddInt(deviceCaps.dwMinSecondarySampleRate);
	fmt.AddInt(deviceCaps.dwMaxSecondarySampleRate);
	LogMessage("   secondary sample rate:     %1; to %2;", fmt);

	LogInt("   primary buffers:           %1;", deviceCaps.dwPrimaryBuffers);
	LogInt("   max hardware mixing:       %1;", deviceCaps.dwMaxHwMixingAllBuffers);
	LogInt("   max static buffers:        %1;", deviceCaps.dwMaxHwMixingStaticBuffers);
	LogInt("   max streaming buffers:     %1;", deviceCaps.dwMaxHwMixingStreamingBuffers);
	LogInt("   free mixing buffers:       %1;", deviceCaps.dwFreeHwMixingAllBuffers);
	LogInt("   free static buffers:       %1;", deviceCaps.dwFreeHwMixingStaticBuffers);
	LogInt("   free streaming buffers:    %1;", deviceCaps.dwFreeHwMixingStreamingBuffers);
	LogInt("   max 3D buffers:            %1;", deviceCaps.dwMaxHw3DAllBuffers);
	LogInt("   max 3D static buffers:     %1;", deviceCaps.dwMaxHw3DStaticBuffers);
	LogInt("   max 3D streaming buffers:  %1;", deviceCaps.dwMaxHw3DStreamingBuffers);
	LogInt("   free 3D buffers:           %1;", deviceCaps.dwFreeHw3DAllBuffers);
	LogInt("   free 3D static buffers:    %1;", deviceCaps.dwFreeHw3DStaticBuffers);
	LogInt("   free 3D streaming buffers: %1;", deviceCaps.dwFreeHw3DStreamingBuffers);
	LogInt("   total hardware memory:     %1;", deviceCaps.dwTotalHwMemBytes);
	LogInt("   free hardware memory:      %1;", deviceCaps.dwFreeHwMemBytes);
	LogInt("   largest contiguous chunk:  %1;", deviceCaps.dwMaxContigFreeHwMemBytes);
	LogInt("   unlocked transfer rate:    %1; K", deviceCaps.dwUnlockTransferRateHwBuffers);
	LogInt("   CPU processing overhead:   %1;%%", deviceCaps.dwPlayCpuOverheadSwBuffers);
}


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzSoundDriver::QzSoundDriver(void)
	:	m_pContext(NULL)
{
	m_pContext = new DirectSoundContext_t;

	memset(m_pContext, 0, sizeof(DirectSoundContext_t));
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzSoundDriver::~QzSoundDriver(void)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	SafeRelease(pContext->pBuffer);
	SafeRelease(pContext->pDxSound);

	SafeDelete(pContext);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Init()
//
//	Initializes DirectSound, creates the primary buffer for output, creates
//	a normal sound buffer for containing queued audio data, and sets up
//	everything that will be needed to Start() playback.
//
bool QzSoundDriver::Init(U32 sampleRate)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	// Protect against multiple calls to Init(), without having first called
	// Stop() to clean up resources.
	SafeRelease(pContext->pBuffer);
	SafeRelease(pContext->pDxSound);

	m_SampleRate = sampleRate;

	// Enumerate all of the available sound drivers for informational
	// purposes.  This is not necessary, but recording the driver
	// information to the log file may be useful if there are problems.
	DirectSoundEnumerate(DirectSoundEnum, reinterpret_cast<void*>(this));

	HRESULT hr = S_OK;

	// Create the DirectSound object required for buffer allocation.
	hr = DirectSoundCreate8(NULL, &(pContext->pDxSound), NULL);
	if (FAILED(hr)) {
		PrintHRESULT("Init() could not create DX Sound", hr);
		return false;
	}

	// Log the device capabilities.  This info may be useful if
	// there are problems with audio output on certain devices.
	CheckCaps(pContext->pDxSound, pContext->DeviceCaps, true);

	// If we're running in normal window (QzMainWin.cpp), this
	// will be the handle where graphics are being rendered.
	HWND hWindow = g_hWindow;

	// However, if hWindow is not defined, assume we're running
	// from a console app, so we can fetch that window handle.
	if (NULL == hWindow) {
		hWindow = GetConsoleWindow();
	}

	// Set the cooperative level.  We need to use PRIORITY level
	// so we can call SetFormat() on the primary mixing buffer.
	hr = pContext->pDxSound->SetCooperativeLevel(hWindow, DSSCL_PRIORITY);
	if (FAILED(hr)) {
		PrintHRESULT("Init() could not set coop level", hr);
		return false;
	}

	// Fill in a struct that defines the primary mixing buffer.
	// Although we won't be directly writing to this buffer, we
	// do need to access it to set the sampling rate (otherwise
	// our audio data may be resampled on playback, which can
	// reduce quality).
	DSBUFFERDESC desc;
	SafeZeroVar(desc);
	desc.dwSize        = sizeof(DSBUFFERDESC);
	desc.dwFlags       = DSBCAPS_PRIMARYBUFFER;
	desc.dwBufferBytes = 0;
	desc.lpwfxFormat   = 0;

	IDirectSoundBuffer *pPrimaryBuffer = NULL;

	hr = pContext->pDxSound->CreateSoundBuffer(&desc, &pPrimaryBuffer, 0);
	if (FAILED(hr)) {
		PrintHRESULT("Init() could not allocate primary buffer", hr);
		return false;
	}

	// Now fill in a WAV struct that defines the formatting of the
	// audio data as 16-bit stereo PCM.
	WAVEFORMATEX format;
	SafeZeroVar(format);
	format.wFormatTag      = WAVE_FORMAT_PCM;
	format.nChannels       = c_AudioChannelCount;
	format.nSamplesPerSec  = m_SampleRate;
	format.nBlockAlign     = 2 * c_AudioChannelCount;
	format.nAvgBytesPerSec = format.nSamplesPerSec * U32(format.nBlockAlign);
	format.wBitsPerSample  = c_SampleBitDepth;

	// Now we SetFormat() on the primary buffer, which sets the audio
	// sample rate that will be used for audio output.
	hr = pPrimaryBuffer->SetFormat(&format);
	if (FAILED(hr)) {
		PrintHRESULT("Init() could not set primary buffer format", hr);
		return false;
	}

	// Release the reference to the primary buffer.
	// We won't be needing it again.
	SafeRelease(pPrimaryBuffer);

	UtfFormat fmt;
	fmt.AddInt(m_SampleRate);
	LogMessage("DirectSound initialized at %1; Hz", fmt);

	// The context info contains the current position at which we will
	// start writing data.  Note that we're setting WriteOffset and
	// BytesRemaining as if that many samples of silence exist at the
	// start of the buffer.  That gives us this amount of playback time
	// after starting before we must write data into the buffer, or
	// risk glitching the audio output and screwing up the write
	// position.
	pContext->BufferSize     = m_SampleRate * c_BytesPerSlice;
	pContext->Position       = 0;
	pContext->WriteOffset    = c_AudioBufferCount * c_BytesPerBuffer;
	pContext->BytesRemaining = c_AudioBufferCount * c_BytesPerBuffer;

	// The exact same buffer settings are used to create the buffer
	// into which we will be writing audio data.  (I'm manually refilling
	// this struct, since I've had problems in the past with Win32 calls
	// changing the contents of the WAV struct.  It doesn't seem to
	// happen here, but once bitten...).
	SafeZeroVar(format);
	format.wFormatTag      = WAVE_FORMAT_PCM;
	format.nChannels       = c_AudioChannelCount;
	format.nSamplesPerSec  = m_SampleRate;
	format.nBlockAlign     = 2 * c_AudioChannelCount;
	format.nAvgBytesPerSec = format.nSamplesPerSec * U32(format.nBlockAlign);
	format.wBitsPerSample  = c_SampleBitDepth;

	// Set up a DSBUFFERDESC structure.  Note that this has different
	// settings than for the primary buffer.  Primarily, we need to be
	// able to get the current playback position so we can write into
	// the buffer as it is being played.
	SafeZeroVar(desc);
	desc.dwSize			= sizeof(DSBUFFERDESC);
	desc.dwFlags		= DSBCAPS_GETCURRENTPOSITION2;
	desc.dwBufferBytes	= format.nAvgBytesPerSec;
	desc.lpwfxFormat	= &format;

	// Set this flag to make sound keep playing when app does not have
	// focus.  Otherwise sound becomes inaudible (but continues to be
	// processed) when mouse/keyboard focus switches to another window.
	desc.dwFlags |= DSBCAPS_GLOBALFOCUS;

	// This creates a basic DirectSound buffer.
	IDirectSoundBuffer *pBuffer = NULL;
	hr = pContext->pDxSound->CreateSoundBuffer(&desc, &pBuffer, NULL);
	if (FAILED(hr)) {
		PrintHRESULT("CreateSoundBuffer() failed", hr);
		return false;
	}

	// Now QI for a DirectSound 8 sound buffer.  This interface
	// will allow us to call GetCurrentPosition() on the buffer.
	hr = pBuffer->QueryInterface(g_SoundBuffer8Guid, (void**)&(pContext->pBuffer));

	// Release the reference to the basic sound buffer interface.
	// We will be using the DS 8 interface from here on out.
	SafeRelease(pBuffer);

	if (FAILED(hr)) {
		PrintHRESULT("QI() failed", hr);
		return false;
	}

	S16* pAdr      = NULL;
	U32  byteCount = 0;

	// Lock the buffer so we can zero out the entire thing.
	hr = pContext->pBuffer->Lock(0, pContext->BufferSize, (void**)&pAdr, &byteCount, NULL, NULL, DSBLOCK_ENTIREBUFFER);

	// It is most unlikely that the buffer could be lost if we're still
	// initializing the app, but try to restore it if we can.
	if (DSERR_BUFFERLOST == hr) {
		pContext->pBuffer->Restore();

		hr = pContext->pBuffer->Lock(0, pContext->BufferSize, (void**)&pAdr, &byteCount, NULL, NULL, DSBLOCK_ENTIREBUFFER);
	}

	if (FAILED(hr)) {
		PrintHRESULT("Lock() failed", hr);
		return false;
	}

	// Zero-out the buffer so it starts off outputting silence.
	memset(pAdr, 0, pContext->BufferSize);

	hr = pContext->pBuffer->Unlock(pAdr, byteCount, NULL, 0);
	if (FAILED(hr)) {
		PrintHRESULT("Unlock() failed", hr);
		return false;
	}

	LogMessage("DirectSound driver Init() completed");

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Start()
//
bool QzSoundDriver::Start(void)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	if ((NULL == pContext->pDxSound) || (NULL == pContext->pBuffer)) {
		LogErrorMessage("Start(): DS not initialized");
		return false;
	}

	HRESULT hr = S_OK;

	// Set playback to start from the beginning of the buffer.
	hr = pContext->pBuffer->SetCurrentPosition(0);

	if (FAILED(hr)) {
		PrintHRESULT("Start(): SetCurrentPosition() failed", hr);
		return false;
	}

	// Start playback to loop infinitely over the contents of the buffer.
	hr = pContext->pBuffer->Play(0, 0, DSBPLAY_LOOPING);

	if (FAILED(hr)) {
		PrintHRESULT("Start(): Play() failed", hr);
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Stop()
//
bool QzSoundDriver::Stop(void)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	if ((NULL == pContext->pDxSound) || (NULL == pContext->pBuffer)) {
		LogErrorMessage("Stop(): DS not initialized");
		return false;
	}

	HRESULT hr = S_OK;

	hr = pContext->pBuffer->Stop();

	if (FAILED(hr)) {
		PrintHRESULT("Stop() failed", hr);
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MinWriteCount()
//
//	What are the minimum number of slices that have to be written at one
//	time?
//
//	Since we can write directly into the audio ring buffer, we can write as
//	little as 1 slice (though generally the audio data will be consumed out
//	of the buffer in chunks of 32 or 64 slices, depending on audio hardware).
//
U32 QzSoundDriver::MinWriteCount(void)
{
	return 1;
}


/////////////////////////////////////////////////////////////////////////////
//
//	FreeBufferCount()
//
//	Report the number of free buffers that need to be filled.
//
//	For DirectSound, always return 1.  There is only one buffer.  But more
//	importantly, if the audio loop were to call UpdatePosition multiple
//	times, it would possibly get different numbers on some calls, even though
//	a second call might only report a handful of samples that would need to
//	be processed.
//
//	The app should only call UpdatePosition() once, then either sleep or
//	otherwise wait for an event to wake it up.
//
U32 QzSoundDriver::FreeBufferCount(void)
{
	return 1;
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdatePosition()
//
//	Returns the number of slices that must be written into the buffer to
//	maintain at least "lookahead" number of slices.
//
U32 QzSoundDriver::UpdatePosition(U32 lookahead)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	U32 write = 0;

	// Get the "write" position.  This is where it is safe to start writing
	// audio data.  Assume that all data in the buffer before this point
	// has been consumed, and plan to start writing data somewhere after
	// this position.
	HRESULT hr = pContext->pBuffer->GetCurrentPosition(NULL, &write);

	if (FAILED(hr)) {
		PrintHRESULT("GetCurrentPosition() failed", hr);
		return 0;
	}

	// Ring buffer logic to figure out how much data has been consumed
	// since the last time we updated the position information.
	U32 consumed = (pContext->BufferSize + write - pContext->Position) % pContext->BufferSize;

	pContext->Position = write;

	// How much valid data is stored in the buffer?
	if (consumed >= pContext->BytesRemaining) {
		// This is the bad condition: we haven't kept up, and the hardware
		// has consumed all of the data we've given it.  We can try to
		// write more data, but we've already had an audio glitch at this
		// point.  And by the time we write more data into the buffer,
		// the hardware will almost certain advanced beyond this new
		// write position, so another glitch will happen then.  Hopefully
		// after this point we can return to keeping ahead of the hardware.
		//
		// If not, we might consider stopping audio, clearing out the whole
		// buffer, then restarting with a long duration of silent data in
		// the buffer.  That will give higher-level code time to queue up
		// more audio data and once again try to keep in advance of where
		// the hardware is reading from the buffer.
		//
		pContext->BytesRemaining = 0;
		pContext->WriteOffset    = write;
	}
	else {
		pContext->BytesRemaining -= consumed;
	}

	// How many slices are currently sitting in the buffer.
	U32 sliceCount = pContext->BytesRemaining / c_BytesPerSlice;

	// If there are not at least "lookahead" number of slices, return the
	// number of slices need to keep "lookahead" slices in advance of what
	// the hardware is processing.
	if (sliceCount < lookahead) {
		return lookahead - sliceCount;
	}

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	WriteToBuffer()
//
bool QzSoundDriver::WriteToBuffer(S16 *pData, U32 sliceCount)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	if (NULL == pContext->pBuffer) {
		return false;
	}

	U32  size1     = 0;
	U32  size2     = 0;
	U08* ptr1      = NULL;
	U08* ptr2      = NULL;
	U32  byteCount = sliceCount * c_BytesPerSlice;

	// Lock a range of memory within the buffer, starting from where we last
	// wrote data, through to the end of where we are going to be writing
	// data this time around.  Since this is a ring buffer, we get back two
	// size values and two pointers.  We only need the second pair of values
	// if the data wraps around the end of the buffer.
	HRESULT hr = pContext->pBuffer->Lock(pContext->WriteOffset, byteCount, (void**)&ptr1, &size1, (void**)&ptr2, &size2, 0);

	if (DSERR_BUFFERLOST == hr) {
		hr = pContext->pBuffer->Restore();

		// NOTE: It is possible for the restore to fail if the app window
		// does not have focus.

		if (FAILED(hr)) {
			PrintHRESULT("WriteToBuffer Restore() failed", hr);
			return false;
		}

		hr = pContext->pBuffer->Lock(pContext->WriteOffset, byteCount, (void**)&ptr1, &size1, (void**)&ptr2, &size2, 0);
	}

	if (FAILED(hr)) {
		PrintHRESULT("WriteToBuffer Lock() failed\n", hr);
		return false;
	}

	// Copy the first segment of data.  If this is in the middle of the
	// buffer, we're done.
	if (NULL != ptr1) {
		memcpy(ptr1, pData, size1);
	}

	// However, if the data range wraps around the end of the buffer,
	// we need to copy the rest of the data to the second memory range,
	// which will start at the beginning of the buffer.
	if (NULL != ptr2) {
		memcpy(ptr2, reinterpret_cast<U08*>(pData) + size1, size2);
	}

	hr = pContext->pBuffer->Unlock(ptr1, size1, ptr2, size2);
	if (FAILED(hr)) {
		PrintHRESULT("WriteToBuffer Unlock() failed\n", hr);
	}

	// Update the write position, keeping in mind that this is a ring buffer.
	pContext->WriteOffset     = (pContext->WriteOffset + byteCount) % pContext->BufferSize;
	pContext->BytesRemaining += byteCount;

	return true;
}

/*
/////////////////////////////////////////////////////////////////////////////
//
//	ZeroOutBuffer()
//
bool QzSoundDriver::ZeroOutBuffer(U32 sliceCount)
{
	DirectSoundContext_t *pContext = reinterpret_cast<DirectSoundContext_t*>(m_pContext);

	if (NULL == pContext->pBuffer) {
		return false;
	}

	U32  size1     = 0;
	U32  size2     = 0;
	U08* ptr1      = NULL;
	U08* ptr2      = NULL;
	U32  byteCount = sliceCount * c_BytesPerSlice;

	HRESULT hr = pContext->pBuffer->Lock(pContext->WriteOffset, byteCount, (void**)&ptr1, &size1, (void**)&ptr2, &size2, 0);

	if (DSERR_BUFFERLOST == hr) {
		pContext->pBuffer->Restore();

		// NOTE: It is possible for the restore to fail if the app window
		// does not have focus.

		if (FAILED(hr)) {
			PrintHRESULT("WriteToBuffer Restore() failed", hr);
			return false;
		}

		hr = pContext->pBuffer->Lock(pContext->WriteOffset, byteCount, (void**)&ptr1, &size1, (void**)&ptr2, &size2, 0);
	}

	if (FAILED(hr)) {
		PrintHRESULT("ZeroOutBuffer Lock() failed", hr);
		return false;
	}

	if (NULL != ptr1) {
		memset(ptr1, 0, size1);
	}

	if (NULL != ptr2) {
		memset(ptr2, 0, size2);
	}

	hr = pContext->pBuffer->Unlock(ptr1, size1, ptr2, size2);
	if (FAILED(hr)) {
		PrintHRESULT("ZeroOutBuffer Unlock() failed", hr);
	}

	pContext->WriteOffset     = (pContext->WriteOffset + byteCount) % pContext->BufferSize;
	pContext->BytesRemaining += byteCount;

	return true;
}
*/





