/////////////////////////////////////////////////////////////////////////////
//
//	File: QzSoundDriverMac.cpp
//
//	$Header: /Projects/Qz/QzSoundDriverMac.cpp  1  2009/9/9 9:01:57a  Lee $
//
//
//	This implements the Mac version of a sound player, using the AudioQueue
//	functionality to feed 16-bit PCM audio to the speakers.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzSoundDriver.h"
#include <CoreAudio/CoreAudio.h>
#include <AudioToolbox/AudioQueue.h>


// This struct is only defined within this file.  It contains all of the
// platform-specific variables needed to control the audio output.  Every
// method that references this data will need to cast the void* m_pContext
// pointer back to this type before it can access the variables.
struct AudioStreamContext_t
{
	AudioStreamBasicDescription DataFormat;
	AudioQueueRef               Queue;
	AudioQueueBufferRef         Buffers[c_AudioBufferCount];
	volatile S32                FreeBufferCount;
	U32                         NextFreeBuffer;
};


/////////////////////////////////////////////////////////////////////////////
//
//	FreeContext()
//
//	This really should be a class method, but that would violate the platform
//	independent intent of the common header file.
//
//	This will free up any buffers or other resources required to drive the
//	AudioQueue functions.
//
static void FreeContext(AudioStreamContext_t *pContext)
{
	for (U32 i = 0; i < c_AudioBufferCount; ++i) {
		if (NULL != pContext->Buffers[i]) {
			AudioQueueFreeBuffer(pContext->Queue, pContext->Buffers[i]);
			pContext->Buffers[i] = NULL;
		}
	}

	if (NULL != pContext->Queue) {
		AudioQueueDispose(pContext->Queue, true);
		pContext->Queue = NULL;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	AudioCallback()
//
//	Callback handler required by AudioQueue to notify the app when each
//	buffer has been consumed.  We can safely ignore the pQueue and pBuffer
//	pointers, since we are not going to try refilling the buffer within the
//	callback (even though that is how the fragments of example code in the
//	docs indicate it should be done).
//
//	All we need to do here is increment a counter so the main thread will
//	know how many empty buffers are waiting to be filled.
//
static void AudioCallback(void *pContext, AudioQueueRef pQueue, AudioQueueBufferRef pBuffer)
{
	reinterpret_cast<QzSoundDriver*>(pContext)->IncFreeCount();
}



/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzSoundDriver::QzSoundDriver(void)
	:	m_pContext(NULL),
		m_SampleRate(0)
{
	m_pContext = new AudioStreamContext_t;

	memset(m_pContext, 0, sizeof(AudioStreamContext_t));
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzSoundDriver::~QzSoundDriver(void)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	FreeContext(pContext);

	SafeDelete(pContext);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Init()
//
bool QzSoundDriver::Init(U32 sampleRate)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	// Protect against multiple calls to Init().
	FreeContext(pContext);

	pContext->DataFormat.mSampleRate       = sampleRate;
	pContext->DataFormat.mFormatID         = kAudioFormatLinearPCM;
	pContext->DataFormat.mFormatFlags      = kAudioFormatFlagIsSignedInteger | kAudioFormatFlagIsPacked;
	pContext->DataFormat.mBytesPerPacket   = c_BytesPerSlice;
	pContext->DataFormat.mFramesPerPacket  = 1;
	pContext->DataFormat.mBytesPerFrame    = c_BytesPerSlice;
	pContext->DataFormat.mChannelsPerFrame = c_AudioChannelCount;
	pContext->DataFormat.mBitsPerChannel   = c_SampleBitDepth;
	pContext->DataFormat.mReserved         = 0;

	m_SampleRate = sampleRate;

	// Create the audio queue that will be used to manage the array of audio
	// buffers used to queue samples.
	OSStatus status = AudioQueueNewOutput(&(pContext->DataFormat), AudioCallback, this, NULL, NULL, 0, &(pContext->Queue));

	if (noErr != status) {
		UtfFormat fmt;
		fmt.AddInt(status);
		LogErrorMessage("AudioQueueNewOutput failed = %1;", fmt);

		return false;
	}

	// Clear these values.  We will pre-queue three buffers of silence before
	// starting output.  When the callback handler is called for the first
	// time, it will indicate that buffer [0] is free, and will increment
	// FreeBufferCount.  This causes the first call to WriteToBuffer() to
	// fill up that buffer, and from there we rely on state logic to keep
	// filling the correct buffers in the correct order.
	pContext->FreeBufferCount = 0;
	pContext->NextFreeBuffer  = 0;

	// Allocate the three buffers we will be using, fill each with silence
	// (all zeroes), and queue them up so they are ready to go once audio
	// output is started.
	for (U32 i = 0; i < c_AudioBufferCount; ++i) {
		status = AudioQueueAllocateBuffer(pContext->Queue, c_BytesPerBuffer, &(pContext->Buffers[i]));
		if (noErr != status) {
			UtfFormat fmt;
			fmt.AddInt(status);
			LogErrorMessage("AudioQueueAllocateBuffer failed = %1;", fmt);

			return false;
		}

		pContext->Buffers[i]->mAudioDataByteSize = c_BytesPerBuffer;
		memset(pContext->Buffers[i]->mAudioData, 0, c_BytesPerBuffer);

		status = AudioQueueEnqueueBuffer(pContext->Queue, pContext->Buffers[i], 0, NULL);

		if (noErr != status) {
			UtfFormat fmt;
			fmt.AddInt(status);
			LogErrorMessage("AudioQueueEnqueueBuffer failed = %1;", fmt);

			return false;
		}
	}

	// Prime the pump.  This will "decode" the PCM data.  However, since the
	// data is already PCM, this really doesn't do anything, and audio will
	// start up and play without this call.  But the docs recommend making
	// this call, and since someone may change this code to take non-PCM
	// audio data, it's a good idea to keep this here for completeness.
	status = AudioQueuePrime(pContext->Queue, 0, NULL);

	if (noErr != status) {
		UtfFormat fmt;
		fmt.AddInt(status);
		LogErrorMessage("AudioQueuePrime failed = %1;", fmt);

		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Start()
//
//	This makes AudioQueue begin playing the first buffer of silent data,
//	kick-starting the state machine buffer sequencing logic.
//
bool QzSoundDriver::Start(void)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	OSStatus status = noErr;

	if (NULL != pContext->Queue) {
		status = AudioQueueStart(pContext->Queue, NULL);

		if (noErr != status) {
			UtfFormat fmt;
			fmt.AddInt(status);
			LogErrorMessage("AudioQueueStart failed = %1;", fmt);

			return false;
		}
	}
	else {
		LogErrorMessage("AudioQueueStart failed: AudioQueue pointer is NULL");
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Stop()
//
bool QzSoundDriver::Stop(void)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	OSStatus status = noErr;

	if (NULL != pContext->Queue) {
		status = AudioQueueStop(pContext->Queue, true);

		if (noErr != status) {
			UtfFormat fmt;
			fmt.AddInt(status);
			LogErrorMessage("AudioQueueStop failed = %1;", fmt);

			return false;
		}
	}
	// Do not display an error if the queue is NULL.  We will explicitly call
	// Stop() from the destructor to assure that audio is stopped and no
	// resources are being leaked.

	FreeContext(pContext);

	return (noErr == status);
}


/////////////////////////////////////////////////////////////////////////////
//
//	MinWriteCount()
//
//	What are the minimum number of slices that have to be written at one
//	time?
//
//	Since we're using fixed-size buffers, it is important for the caller
//	to know how much is required per call to WriteToBuffer().
//
U32 QzSoundDriver::MinWriteCount(void)
{
	return c_SlicesPerBuffer;
}


/////////////////////////////////////////////////////////////////////////////
//
//	FreeBufferCount()
//
//	Report the number of free buffers that need to be filled.
//
//	For the Mac, we just need to return the counter that get incremented by
//	the callback function.
//
U32 QzSoundDriver::FreeBufferCount(void)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	return pContext->FreeBufferCount;
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdatePosition()
//
//	Reports how many samples can be given to WriteToBuffer().
//
//	Since we're using fixed-size buffers, we must return c_SlicesPerBuffer
//	so the caller knows exactly how much data is expected by WriteToBuffer().
//
//	If there are no buffers free, return zero so the caller does not attempt
//	to queue up more audio data.
//
U32 QzSoundDriver::UpdatePosition(U32 /*lookahead*/)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	return (0 == pContext->FreeBufferCount) ? 0 : c_SlicesPerBuffer;
}


/////////////////////////////////////////////////////////////////////////////
//
//	WriteToBuffer()
//
//	Write an array of slices into the next available audio buffer, and queue
//	it for playback.
//
//	Since we're using an array of fixed-sized audio buffers, this code
//	assumes that we are being given exactly the number of slices required
//	to fill the buffer.
//
bool QzSoundDriver::WriteToBuffer(S16 *pData, U32 sliceCount)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	OSStatus status = noErr;

	// We need at least one free buffer.  The caller should have used
	// UpdatePosition() to see if there is any buffer space available
	// before calling WriteToBuffer().
	if (0 == pContext->FreeBufferCount) {
		return false;
	}

	// We can only accept a full buffer of audio at a time.
	// (Actually, we could accept less, but that would make this
	// control logic more elaborate.)
	if (c_SlicesPerBuffer != sliceCount) {
		UtfFormat fmt;
		fmt.AddInt(sliceCount);
		fmt.AddInt(c_SlicesPerBuffer);
		LogErrorMessage("WriteToBuffer failed, invalid slice count: %1; != %2;", fmt);

		return false;
	}

	U32 index = pContext->NextFreeBuffer;

	// Blit the data into the next available buffer.
	pContext->Buffers[index]->mAudioDataByteSize = c_BytesPerBuffer;
	memcpy(pContext->Buffers[index]->mAudioData, pData, c_BytesPerBuffer);

	// Queue the buffer for playback.
	status = AudioQueueEnqueueBuffer(pContext->Queue, pContext->Buffers[index], 0, NULL);

	if (noErr != status) {
		UtfFormat fmt;
		fmt.AddInt(status);
		LogErrorMessage("AudioQueueEnqueueBuffer failed = %1;", fmt);

		return false;
	}

	// Decrement the count of empty buffers and advance NextFreeBuffer
	// around to index of the next buffer in the three-buffer ring.
	QzThreadSafeDecrement((S32*)&(pContext->FreeBufferCount));
	pContext->NextFreeBuffer = (pContext->NextFreeBuffer + 1) % c_AudioBufferCount;

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IncFreeCount()
//
//	This is only needed for the callback function to increment the count of
//	empty buffers.  Note that the DirectSound version of this class does not
//	have this method implemented, since it does not use callbacks.
//
void QzSoundDriver::IncFreeCount(void)
{
	AudioStreamContext_t *pContext = reinterpret_cast<AudioStreamContext_t*>(m_pContext);

	QzThreadSafeIncrement((S32*)&(pContext->FreeBufferCount));
}



