/////////////////////////////////////////////////////////////////////////////
//
//	File: QzMatrix4x4.cpp
//
//	$Header: /TS/TsGui/QzMatrix4x4.cpp  41  2009/9/7 3:53:38p  Lee $
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzMatrix4x4.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	IsEqual()
//
bool QzMatrix4x4::IsEqual(const QzMatrix4x4 &src)
{
	for (U32 i = 0; i < 16; ++i) {
		if (false == QzEqualFloats20(m_Matrix.cFlat[i], src.m_Matrix.cFlat[i])) {
			return false;
		}
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	IsIdentity()
//
bool QzMatrix4x4::IsIdentity(void)
{
	float *f = m_Matrix.cFlat;

	return (QzEqualFloats20(1.0f, f[0]) &&
			QzEqualFloats20(1.0f, f[5]) &&
			QzEqualFloats20(1.0f, f[10]) &&
			QzEqualFloats20(1.0f, f[15]) &&
			QzFloatIsZero20(f[1]) &&
			QzFloatIsZero20(f[2]) &&
			QzFloatIsZero20(f[3]) &&
			QzFloatIsZero20(f[4]) &&
			QzFloatIsZero20(f[6]) &&
			QzFloatIsZero20(f[7]) &&
			QzFloatIsZero20(f[8]) &&
			QzFloatIsZero20(f[9]) &&
			QzFloatIsZero20(f[11]) &&
			QzFloatIsZero20(f[12]) &&
			QzFloatIsZero20(f[13]) &&
			QzFloatIsZero20(f[14]));
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeIdentity()
//
void QzMatrix4x4::MakeIdentity(void)
{
	MakeZero();

	m_Matrix.c4x4[0][0] = 1.0f;
	m_Matrix.c4x4[1][1] = 1.0f;
	m_Matrix.c4x4[2][2] = 1.0f;
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeWindow()
//
//	Sets up a projection matrix so that it maps normal image coords into the
//	window, with (0,0) as the upper left corner.
//
void QzMatrix4x4::MakeWindow(QzRect rect, float nearZ, float farZ)
{
	//
	// WARNING: Some logic below uses (left - right) and (right - left)
	// depending upon what the resulting sign should be.
	//

	m_Matrix.c4x4[0][0] = 2.0f / float(rect.m_Right - rect.m_Left);
	m_Matrix.c4x4[0][1] = 0.0f;
	m_Matrix.c4x4[0][2] = 0.0f;
	m_Matrix.c4x4[0][3] = 0.0f;
	m_Matrix.c4x4[1][0] = 0.0f;
	m_Matrix.c4x4[1][1] = 2.0f / float(rect.m_Top - rect.m_Bottom);
	m_Matrix.c4x4[1][2] = 0.0f;
	m_Matrix.c4x4[1][3] = 0.0f;
	m_Matrix.c4x4[2][0] = 0.0f;
	m_Matrix.c4x4[2][1] = 0.0f;
	m_Matrix.c4x4[2][2] = 1.0f / (farZ - nearZ);
	m_Matrix.c4x4[2][3] = 0.0f;
	m_Matrix.c4x4[3][0] = float(rect.m_Left + rect.m_Right) / float(rect.m_Left - rect.m_Right);
	m_Matrix.c4x4[3][1] = float(rect.m_Top + rect.m_Bottom) / float(rect.m_Bottom - rect.m_Top);
	m_Matrix.c4x4[3][2] = nearZ / (nearZ - farZ);
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateX()
//
void QzMatrix4x4::MakeRotateX(float angle)
{
	MakeZero();

	float sinTheta = sinf(angle);
	float cosTheta = cosf(angle);

	m_Matrix.c4x4[0][0] =  1.0f;
	m_Matrix.c4x4[1][1] =  cosTheta;
	m_Matrix.c4x4[1][2] =  sinTheta;
	m_Matrix.c4x4[2][1] = -sinTheta;
	m_Matrix.c4x4[2][2] =  cosTheta;
	m_Matrix.c4x4[3][3] =  1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateY()
//
void QzMatrix4x4::MakeRotateY(float angle)
{
	MakeZero();

	float sinTheta = sinf(angle);
	float cosTheta = cosf(angle);

	m_Matrix.c4x4[0][0] =  cosTheta;
	m_Matrix.c4x4[0][2] = -sinTheta;
	m_Matrix.c4x4[1][1] =  1.0f;
	m_Matrix.c4x4[2][0] =  sinTheta;
	m_Matrix.c4x4[2][2] =  cosTheta;
	m_Matrix.c4x4[3][3] =  1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateZ()
//
void QzMatrix4x4::MakeRotateZ(float angle)
{
	MakeZero();

	float sinTheta = sinf(angle);
	float cosTheta = cosf(angle);

	m_Matrix.c4x4[0][0] =  cosTheta;
	m_Matrix.c4x4[0][1] =  sinTheta;
	m_Matrix.c4x4[1][0] = -sinTheta;
	m_Matrix.c4x4[1][1] =  cosTheta;
	m_Matrix.c4x4[2][2] =  1.0f;
	m_Matrix.c4x4[3][3] =  1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRotateArbitraryAxis()
//
//	This assumes that the axis vector (x,y,z) is normalized.
//
void QzMatrix4x4::MakeRotateArbitraryAxis(float x, float y, float z, float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);
	float t = 1.0f - c;

	m_Matrix.c4x4[0][0] = t * x * x + c;
	m_Matrix.c4x4[0][1] = t * y * x + s * z;
	m_Matrix.c4x4[0][2] = t * z * x - s * y;
	m_Matrix.c4x4[0][3] = 0.0f;

	m_Matrix.c4x4[1][0] = t * x * y - s * z;
	m_Matrix.c4x4[1][1] = t * y * y + c;
	m_Matrix.c4x4[1][2] = t * z * y + s * x;
	m_Matrix.c4x4[1][3] = 0.0f;

	m_Matrix.c4x4[2][0] = t * x * z + s * y;
	m_Matrix.c4x4[2][1] = t * y * z - s * x;
	m_Matrix.c4x4[2][2] = t * z * z + c;
	m_Matrix.c4x4[2][3] = 0.0f;

	m_Matrix.c4x4[3][0] = 0.0f;
	m_Matrix.c4x4[3][1] = 0.0f;
	m_Matrix.c4x4[3][2] = 0.0f;
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeScale()
//
void QzMatrix4x4::MakeScale(float x, float y, float z)
{
	MakeZero();

	m_Matrix.c4x4[0][0] = x;
	m_Matrix.c4x4[1][1] = y;
	m_Matrix.c4x4[2][2] = z;
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeScale()
//
void QzMatrix4x4::MakeScale(float factor)
{
	MakeZero();

	m_Matrix.c4x4[0][0] = factor;
	m_Matrix.c4x4[1][1] = factor;
	m_Matrix.c4x4[2][2] = factor;
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeTranslate()
//
void QzMatrix4x4::MakeTranslate(float x, float y, float z)
{
	MakeIdentity();

	m_Matrix.c4x4[3][0] = x;
	m_Matrix.c4x4[3][1] = y;
	m_Matrix.c4x4[3][2] = z;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeLeftHandedPerspectiveFOV()
//
void QzMatrix4x4::MakeLeftHandedPerspectiveFOV(float fov, float aspectRatio, float nearZ, float farZ)
{
	float scaleY = 1.0f / tanf(fov * 0.5f);
	float scaleX = scaleY / aspectRatio;

	m_Matrix.c4x4[0][0] = scaleX;
	m_Matrix.c4x4[0][1] = 0.0f;
	m_Matrix.c4x4[0][2] = 0.0f;
	m_Matrix.c4x4[0][3] = 0.0f;
	m_Matrix.c4x4[1][0] = 0.0f;
	m_Matrix.c4x4[1][1] = scaleY;
	m_Matrix.c4x4[1][2] = 0.0f;
	m_Matrix.c4x4[1][3] = 0.0f;
	m_Matrix.c4x4[2][0] = 0.0f;
	m_Matrix.c4x4[2][1] = 0.0f;
	m_Matrix.c4x4[2][2] = farZ / (farZ - nearZ);
	m_Matrix.c4x4[2][3] = 1.0f;
	m_Matrix.c4x4[3][0] = 0.0f;
	m_Matrix.c4x4[3][1] = 0.0f;
	m_Matrix.c4x4[3][2] = -nearZ * m_Matrix.c4x4[2][2];
	m_Matrix.c4x4[3][3] = 0.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRightHandedPerspectiveFOV()
//
void QzMatrix4x4::MakeRightHandedPerspectiveFOV(float fov, float aspectRatio, float nearZ, float farZ)
{
	float scaleY = 1.0f / tanf(fov * 0.5f);
	float scaleX = scaleY / aspectRatio;

	m_Matrix.c4x4[0][0] = scaleX;
	m_Matrix.c4x4[0][1] = 0.0f;
	m_Matrix.c4x4[0][2] = 0.0f;
	m_Matrix.c4x4[0][3] = 0.0f;
	m_Matrix.c4x4[1][0] = 0.0f;
	m_Matrix.c4x4[1][1] = scaleY;
	m_Matrix.c4x4[1][2] = 0.0f;
	m_Matrix.c4x4[1][3] = 0.0f;
	m_Matrix.c4x4[2][0] = 0.0f;
	m_Matrix.c4x4[2][1] = 0.0f;
	m_Matrix.c4x4[2][2] = -farZ / (farZ - nearZ);
	m_Matrix.c4x4[2][3] = -1.0f;
	m_Matrix.c4x4[3][0] = 0.0f;
	m_Matrix.c4x4[3][1] = 0.0f;
	m_Matrix.c4x4[3][2] = nearZ * m_Matrix.c4x4[2][2];
	m_Matrix.c4x4[3][3] = 0.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeLeftHandedParallelFOV()
//
void QzMatrix4x4::MakeLeftHandedParallelFOV(float width, float height, float nearZ, float farZ)
{
	m_Matrix.c4x4[0][0] = 2.0f / width;
	m_Matrix.c4x4[0][1] = 0.0f;
	m_Matrix.c4x4[0][2] = 0.0f;
	m_Matrix.c4x4[0][3] = 0.0f;
	m_Matrix.c4x4[1][0] = 0.0f;
	m_Matrix.c4x4[1][1] = 2.0f / height;
	m_Matrix.c4x4[1][2] = 0.0f;
	m_Matrix.c4x4[1][3] = 0.0f;
	m_Matrix.c4x4[2][0] = 0.0f;
	m_Matrix.c4x4[2][1] = 0.0f;
	m_Matrix.c4x4[2][2] = 1.0f / (farZ - nearZ);
	m_Matrix.c4x4[2][3] = 0.0f;
	m_Matrix.c4x4[3][0] = 0.0f;
	m_Matrix.c4x4[3][1] = 0.0f;
	m_Matrix.c4x4[3][2] = nearZ / (nearZ - farZ);
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeLeftHandedParallelFOV()
//
void QzMatrix4x4::MakeLeftHandedParallelFOV(const QzRect &rect, float nearZ, float farZ)
{
	//
	// WARNING: Some logic below uses (left - right) and (right - left)
	// depending upon what the resulting sign should be.
	//

	m_Matrix.c4x4[0][0] = 2.0f / float(rect.m_Right - rect.m_Left);
	m_Matrix.c4x4[0][1] = 0.0f;
	m_Matrix.c4x4[0][2] = 0.0f;
	m_Matrix.c4x4[0][3] = 0.0f;
	m_Matrix.c4x4[1][0] = 0.0f;
	m_Matrix.c4x4[1][1] = 2.0f / float(rect.m_Top - rect.m_Bottom);
	m_Matrix.c4x4[1][2] = 0.0f;
	m_Matrix.c4x4[1][3] = 0.0f;
	m_Matrix.c4x4[2][0] = 0.0f;
	m_Matrix.c4x4[2][1] = 0.0f;
	m_Matrix.c4x4[2][2] = 1.0f / (farZ - nearZ);
	m_Matrix.c4x4[2][3] = 0.0f;
	m_Matrix.c4x4[3][0] = float(rect.m_Left + rect.m_Right) / float(rect.m_Left - rect.m_Right);
	m_Matrix.c4x4[3][1] = float(rect.m_Top + rect.m_Bottom) / float(rect.m_Bottom - rect.m_Top);
	m_Matrix.c4x4[3][2] = nearZ / (nearZ - farZ);
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	MakeRightHandedParallelFOV()
//
void QzMatrix4x4::MakeRightHandedParallelFOV(float width, float height, float nearZ, float farZ)
{
	m_Matrix.c4x4[0][0] = 2.0f / width;
	m_Matrix.c4x4[0][1] = 0.0f;
	m_Matrix.c4x4[0][2] = 0.0f;
	m_Matrix.c4x4[0][3] = 0.0f;
	m_Matrix.c4x4[1][0] = 0.0f;
	m_Matrix.c4x4[1][1] = 2.0f / height;
	m_Matrix.c4x4[1][2] = 0.0f;
	m_Matrix.c4x4[1][3] = 0.0f;
	m_Matrix.c4x4[2][0] = 0.0f;
	m_Matrix.c4x4[2][1] = 0.0f;
	m_Matrix.c4x4[2][2] = 1.0f / (nearZ - farZ);
	m_Matrix.c4x4[2][3] = 0.0f;
	m_Matrix.c4x4[3][0] = 0.0f;
	m_Matrix.c4x4[3][1] = 0.0f;
	m_Matrix.c4x4[3][2] = nearZ / (nearZ - farZ);
	m_Matrix.c4x4[3][3] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotateX()
//
void QzMatrix4x4::PreRotateX(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	float *p = m_Matrix.cFlat;

	for (U32 i = 0; i < 4; ++i) {
		float t = p[4+i];
		float u = p[8+i];
		p[4+i]  = (t * c) + (u * s);
		p[8+i]  = (u * c) - (t * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotateX()
//
void QzMatrix4x4::PostRotateX(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	float *p = m_Matrix.cFlat;

	for (U32 i = 0; i < 4; ++i) {
		float t = p[(i<<2)+1];
		float u = p[(i<<2)+2];
		p[(i<<2)+1] = (t * c) - (u * s);
		p[(i<<2)+2] = (t * s) + (u * c);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotateY()
//
void QzMatrix4x4::PreRotateY(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	float *p = m_Matrix.cFlat;

	for (U32 i = 0; i < 4; ++i) {
		float t = p[  i];
		float u = p[8+i];
		p[  i]  = (t * c) - (u * s);
		p[8+i]  = (u * c) + (t * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotateY()
//
void QzMatrix4x4::PostRotateY(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	float *p = m_Matrix.cFlat;

	for (U32 i = 0; i < 4; ++i) {
		float t = p[(i<<2)  ];
		float u = p[(i<<2)+2];
		p[(i<<2)  ] = (u * s) + (t * c);
		p[(i<<2)+2] = (u * c) - (t * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotateZ()
//
void QzMatrix4x4::PreRotateZ(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	float *p = m_Matrix.cFlat;

	for (U32 i = 0; i < 4; ++i) {
		float t = p[  i];
		float u = p[4+i];
		p[  i]  = (t * c) + (u * s);
		p[4+i]  = (u * c) - (t * s);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotateZ()
//
void QzMatrix4x4::PostRotateZ(float angle)
{
	float c = cosf(angle);
	float s = sinf(angle);

	float *p = m_Matrix.cFlat;

	for (U32 i = 0; i < 4; ++i) {
		float t = p[(i<<2)  ];
		float u = p[(i<<2)+1];
		p[(i<<2)  ] = (t * c) - (u * s);
		p[(i<<2)+1] = (t * s) + (u * c);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreRotate()
//
void QzMatrix4x4::PreRotate(QzQuat &quat)
{
	QzMatrix4x4 temp;
	quat.ConvertToMatrix(temp);
	PreMultiply(temp);
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostRotate()
//
void QzMatrix4x4::PostRotate(QzQuat &quat)
{
	QzMatrix4x4 temp;
	quat.ConvertToMatrix(temp);
	PostMultiply(temp);
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreScale()
//
void QzMatrix4x4::PreScale(float x, float y, float z)
{
	for (U32 i = 0; i < 4; ++i) {
		m_Matrix.c4x4[0][i] *= x;
		m_Matrix.c4x4[1][i] *= y;
		m_Matrix.c4x4[2][i] *= z;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreScale()
//
void QzMatrix4x4::PreScale(float x)
{
	for (U32 i = 0; i < 4; ++i) {
		m_Matrix.c4x4[0][i] *= x;
		m_Matrix.c4x4[1][i] *= x;
		m_Matrix.c4x4[2][i] *= x;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostScale()
//
void QzMatrix4x4::PostScale(float x, float y, float z)
{
	for (U32 i = 0; i < 4; ++i) {
		m_Matrix.c4x4[i][0] *= x;
		m_Matrix.c4x4[i][1] *= y;
		m_Matrix.c4x4[i][2] *= z;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostScale()
//
void QzMatrix4x4::PostScale(float x)
{
	for (U32 i = 0; i < 4; ++i) {
		m_Matrix.c4x4[i][0] *= x;
		m_Matrix.c4x4[i][1] *= x;
		m_Matrix.c4x4[i][2] *= x;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreTranslate()
//
void QzMatrix4x4::PreTranslate(float x, float y, float z)
{
	for (U32 i = 0; i < 4; ++i) {
		m_Matrix.c4x4[3][i] += (x * m_Matrix.c4x4[0][i]) + (y * m_Matrix.c4x4[1][i]) + (z * m_Matrix.c4x4[2][i]);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostTranslate()
//
void QzMatrix4x4::PostTranslate(float x, float y, float z)
{
	for (U32 i = 0; i < 4; ++i) {
		m_Matrix.c4x4[i][0] += m_Matrix.c4x4[i][3] * x;
		m_Matrix.c4x4[i][1] += m_Matrix.c4x4[i][3] * y;
		m_Matrix.c4x4[i][2] += m_Matrix.c4x4[i][3] * z;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreTripleRotate()
//
void QzMatrix4x4::PreTripleRotate(const QzVector rotate)
{
	PreRotateY(rotate.m_Z);
	PreRotateX(rotate.m_X);
	PreRotateY(rotate.m_Y);
}


/////////////////////////////////////////////////////////////////////////////
//
//	ComputeInverseMatrix()
//
//	Find the inverse of the matrix by first computing the adjoint of the
//	original matrix, then dividing by the adjoint.  This is only processing
//	the upper-left 3x3 rotation matrix.  The translation part of the matrix
//	is extracted separately.
//
//	Obviously, this is not a generalized inverse.  It only works correctly on
//	homogeneous transforms, where the last row is the translation and the
//	last column is filled with zeroes.
//
void QzMatrix4x4::ComputeInverseMatrix(QzMatrix4x4 &src)
{
	float *s = src.m_Matrix.cFlat;
	float *d = m_Matrix.cFlat;

	// Compute the adjoint of the matrix.
	d[ 0] = (s[5] * s[10]) - (s[6] * s[ 9]);
	d[ 1] = (s[2] * s[ 9]) - (s[1] * s[10]);
	d[ 2] = (s[1] * s[ 6]) - (s[2] * s[ 5]);
	d[ 3] = 0.0f;
	d[ 4] = (s[6] * s[ 8]) - (s[4] * s[10]);
	d[ 5] = (s[0] * s[10]) - (s[2] * s[ 8]);
	d[ 6] = (s[2] * s[ 4]) - (s[0] * s[ 6]);
	d[ 7] = 0.0f;
	d[ 8] = (s[4] * s[ 9]) - (s[5] * s[ 8]);
	d[ 9] = (s[1] * s[ 8]) - (s[0] * s[ 9]);
	d[10] = (s[0] * s[ 5]) - (s[1] * s[ 4]);
	d[11] = 0.0f;

	// Compute the determinant.  The determinant is derived from more complex
	// values.  But we just derived those values above, so we can reuse them
	// to directly extract the determinant.
	float determinant = (s[0] * d[0]) + (s[1] * d[4]) + (s[2] * d[8]);

	// Divide the adjoint by the determinant to compute the inverse rotation
	// matrix.  Assuming that we're always given a homogeneous transform,
	// the determinant should never be zero.  But always guard against
	// invalid divisions.
	if (false == QzFloatIsZero20(determinant)) {
		determinant = 1.0f / determinant;
		d[ 0] *= determinant;
		d[ 1] *= determinant;
		d[ 2] *= determinant;
		d[ 4] *= determinant;
		d[ 5] *= determinant;
		d[ 6] *= determinant;
		d[ 8] *= determinant;
		d[ 9] *= determinant;
		d[10] *= determinant;
	}

	// Transform against the inverse rotation to extract the inverse of the
	// translation.
	d[12] = -((s[12] * d[0]) + (s[13] * d[4]) + (s[14] * d[ 8]));
	d[13] = -((s[12] * d[1]) + (s[13] * d[5]) + (s[14] * d[ 9]));
	d[14] = -((s[12] * d[2]) + (s[13] * d[6]) + (s[14] * d[10]));
	d[15] = 1.0f;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ComputeNormalMatrix()
//
//	This is almost identical to computing the inverse of the matrix.  The
//	two differences are that for a normal matrix we don't care about the
//	translation, and that we need to invert the transpose of the matrix.
//	So this code will zero out the transform portion of the matrix, and it
//	will normalize the rotation vectors so that unit vectors retain unit
//	length when rotated.
//
void QzMatrix4x4::ComputeNormalMatrix(QzMatrix4x4 &src)
{
	float *s = src.m_Matrix.cFlat;
	float *d = m_Matrix.cFlat;

	// Compute the adjoint of the matrix.
	d[ 0] = (s[5] * s[10]) - (s[6] * s[ 9]);
	d[ 1] = (s[6] * s[ 8]) - (s[4] * s[10]);
	d[ 2] = (s[4] * s[ 9]) - (s[5] * s[ 8]);
	d[ 3] = 0.0f;
	d[ 4] = (s[2] * s[ 9]) - (s[1] * s[10]);
	d[ 5] = (s[0] * s[10]) - (s[2] * s[ 8]);
	d[ 6] = (s[1] * s[ 8]) - (s[0] * s[ 9]);
	d[ 7] = 0.0f;
	d[ 8] = (s[1] * s[ 6]) - (s[2] * s[ 5]);
	d[ 9] = (s[2] * s[ 4]) - (s[0] * s[ 6]);
	d[10] = (s[0] * s[ 5]) - (s[1] * s[ 4]);
	d[11] = 0.0f;
	d[12] = 0.0f;
	d[13] = 0.0f;
	d[14] = 0.0f;
	d[15] = 1.0f;

	// Compute the determinant.  The determinant is derived from more complex
	// values.  But we just derived those values above, so we can reuse them
	// to directly extract the determinant.
	float determinant = (s[0] * d[0]) + (s[1] * d[2]) + (s[2] * d[2]);

	// Divide the adjoint by the determinant to compute the inverse rotation
	// matrix.  Assuming that we're always given a homogeneous transform,
	// the determinant should never be zero.  But always guard against
	// invalid divisions.
	if (false == QzFloatIsZero20(determinant)) {
		determinant = 1.0f / determinant;
		d[ 0] *= determinant;
		d[ 1] *= determinant;
		d[ 2] *= determinant;
		d[ 4] *= determinant;
		d[ 5] *= determinant;
		d[ 6] *= determinant;
		d[ 8] *= determinant;
		d[ 9] *= determinant;
		d[10] *= determinant;
	}

	// Now normalize the rotation vectors (since applying a rotation matrix
	// is effectively applying three dot products, so each of the three
	// vectors should be of unit length).
	//
	// Note that we're using QzEqualFloats10() here, since it's comparing the
	// square of the length.  Values close to 1.0 will be further from 1.0
	// before squaring, so we don't need as much precision on the comparison.
	//

	float len;

	len = (d[0] * d[0]) + (d[4] * d[4]) + (d[8] * d[8]);
	if (false == QzEqualFloats10(1.0f, len)) {
		len = sqrtf(len);
		d[0] /= len;
		d[4] /= len;
		d[8] /= len;
	}

	len = (d[1] * d[1]) + (d[5] * d[5]) + (d[9] * d[9]);
	if (false == QzEqualFloats10(1.0f, len)) {
		len = sqrtf(len);
		d[1] /= len;
		d[5] /= len;
		d[9] /= len;
	}

	len = (d[2] * d[2]) + (d[6] * d[6]) + (d[10] * d[10]);
	if (false == QzEqualFloats10(1.0f, len)) {
		len = sqrtf(len);
		d[ 2] /= len;
		d[ 6] /= len;
		d[10] /= len;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Multiply()
//
void QzMatrix4x4::Multiply(const QzMatrix4x4 &first, const QzMatrix4x4 &second)
{
	const float *m1 = first.m_Matrix.cFlat;
	const float *m2 = second.m_Matrix.cFlat;
	      float *r  = m_Matrix.cFlat;

	for (U32 row = 0; row < 4; ++row) {
		for (U32 col = 0; col < 4; ++col) {
			r[(row<<2)+col] = m1[(row<<2)  ] * m2[col   ]
							+ m1[(row<<2)+1] * m2[col+ 4]
							+ m1[(row<<2)+2] * m2[col+ 8]
							+ m1[(row<<2)+3] * m2[col+12];
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PreMultiply()
//
void QzMatrix4x4::PreMultiply(const QzMatrix4x4 &first)
{
	// Multiplication is destructive, so we need make a duplicate of the
	// original value of m_Matrix for use in computing the new m_Matrix.
	const QzMatrix4x4_t dup = m_Matrix;
	const float*        m1  = first.m_Matrix.cFlat;
	const float*        m2  = dup.cFlat;
	      float*        r   = m_Matrix.cFlat;

	for (U32 row = 0; row < 4; ++row) {
		for (U32 col = 0; col < 4; ++col) {
			r[(row<<2)+col] = m1[(row<<2)  ] * m2[col   ]
							+ m1[(row<<2)+1] * m2[col+ 4]
							+ m1[(row<<2)+2] * m2[col+ 8]
							+ m1[(row<<2)+3] * m2[col+12];
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	PostMultiply()
//
void QzMatrix4x4::PostMultiply(const QzMatrix4x4 &second)
{
	// Multiplication is destructive, so we need make a duplicate of the
	// original value of m_Matrix for use in computing the new m_Matrix.
	const QzMatrix4x4_t dup = m_Matrix;
	const float*        m1  = dup.cFlat;
	const float*        m2  = second.m_Matrix.cFlat;
	      float*        r   = m_Matrix.cFlat;

	for (U32 row = 0; row < 4; ++row) {
		for (U32 col = 0; col < 4; ++col) {
			r[(row<<2)+col] = m1[(row<<2)  ] * m2[col   ]
							+ m1[(row<<2)+1] * m2[col+ 4]
							+ m1[(row<<2)+2] * m2[col+ 8]
							+ m1[(row<<2)+3] * m2[col+12];
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	TransformVector()
//
void QzMatrix4x4::TransformVector(const QzVector vec, QzVector &result)
{
	QzMatrix4x4_t &m = m_Matrix;

	// Minor optimization.  Don't need to multiply against the 4th column of
	// the matrix, since there is no W value with a 3-vector.  And the final
	// row of numbers are all multiplied against 1.0f, so skip that multiply
	// as well.
	result.m_X = (vec.m_X * m.c4x4[0][0]) + (vec.m_Y * m.c4x4[1][0]) + (vec.m_Z * m.c4x4[2][0]) + m.c4x4[3][0];
	result.m_Y = (vec.m_X * m.c4x4[0][1]) + (vec.m_Y * m.c4x4[1][1]) + (vec.m_Z * m.c4x4[2][1]) + m.c4x4[3][1];
	result.m_Z = (vec.m_X * m.c4x4[0][2]) + (vec.m_Y * m.c4x4[1][2]) + (vec.m_Z * m.c4x4[2][2]) + m.c4x4[3][2];
}


/////////////////////////////////////////////////////////////////////////////
//
//	TransformVector()
//
void QzMatrix4x4::TransformVector(const QzVector_t vec, QzVector_t &result)
{
	QzMatrix4x4_t &m = m_Matrix;

	// Minor optimization.  Don't need to multiply against the 4th column of
	// the matrix, since there is no W value with a 3-vector.  And the final
	// row of numbers are all multiplied against 1.0f, so skip that multiply
	// as well.
	result.X = (vec.X * m.c4x4[0][0]) + (vec.Y * m.c4x4[1][0]) + (vec.Z * m.c4x4[2][0]) + m.c4x4[3][0];
	result.Y = (vec.X * m.c4x4[0][1]) + (vec.Y * m.c4x4[1][1]) + (vec.Z * m.c4x4[2][1]) + m.c4x4[3][1];
	result.Z = (vec.X * m.c4x4[0][2]) + (vec.Y * m.c4x4[1][2]) + (vec.Z * m.c4x4[2][2]) + m.c4x4[3][2];
}


/////////////////////////////////////////////////////////////////////////////
//
//	TransformVector()
//
void QzMatrix4x4::TransformVectorW(const QzVector vec, QzVector &result, float &w)
{
	QzMatrix4x4_t &m = m_Matrix;

	// Minor optimization.  Don't need to multiply against the 4th column of
	// the matrix, since there is no W value with a 3-vector.  And the final
	// row of numbers are all multiplied against 1.0f, so skip that multiply
	// as well.
	result.m_X = (vec.m_X * m.c4x4[0][0]) + (vec.m_Y * m.c4x4[1][0]) + (vec.m_Z * m.c4x4[2][0]) + m.c4x4[3][0];
	result.m_Y = (vec.m_X * m.c4x4[0][1]) + (vec.m_Y * m.c4x4[1][1]) + (vec.m_Z * m.c4x4[2][1]) + m.c4x4[3][1];
	result.m_Z = (vec.m_X * m.c4x4[0][2]) + (vec.m_Y * m.c4x4[1][2]) + (vec.m_Z * m.c4x4[2][2]) + m.c4x4[3][2];

	w = m.c4x4[0][3] + m.c4x4[1][3] + m.c4x4[2][3] + m.c4x4[3][3];
}


/////////////////////////////////////////////////////////////////////////////
//
//	TransformNormal()
//
//	This assumes that no scaling has been applied to the matrix.
//
//	In the case of uniform scaling, the resulting normal will have the
//	correct orientation, but the wrong length.  The normal will need to be
//	normalized to make it unit length.
//
//	In the case of non-uniform scaling, this will not work.  The resulting
//	normal will have the wrong orientation.  ComputeNormalMatrix() will be
//	needed to create a valid normal transform matrix.
//
void QzMatrix4x4::TransformNormal(const QzVector vec, QzVector &result)
{
	QzMatrix4x4_t &m = m_Matrix;

	result.m_X = (vec.m_X * m.c4x4[0][0]) + (vec.m_Y * m.c4x4[1][0]) + (vec.m_Z * m.c4x4[2][0]);
	result.m_Y = (vec.m_X * m.c4x4[0][1]) + (vec.m_Y * m.c4x4[1][1]) + (vec.m_Z * m.c4x4[2][1]);
	result.m_Z = (vec.m_X * m.c4x4[0][2]) + (vec.m_Y * m.c4x4[1][2]) + (vec.m_Z * m.c4x4[2][2]);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Transpose()
//
void QzMatrix4x4::Transpose(void)
{
	float *p = m_Matrix.cFlat;

	Swap(p[ 1], p[ 4]);
	Swap(p[ 2], p[ 8]);
	Swap(p[ 3], p[12]);
	Swap(p[ 6], p[ 9]);
	Swap(p[ 7], p[13]);
	Swap(p[11], p[14]);
}


/////////////////////////////////////////////////////////////////////////////
//
//	LeftHandedToRightHanded()
//
void QzMatrix4x4::LeftHandedToRightHanded(const QzMatrix4x4 &src)
{
	m_Matrix.c4x4[0][0] =  src.m_Matrix.c4x4[0][0];
	m_Matrix.c4x4[0][1] =  src.m_Matrix.c4x4[0][1];
	m_Matrix.c4x4[0][2] = -src.m_Matrix.c4x4[0][2];
	m_Matrix.c4x4[0][3] =  src.m_Matrix.c4x4[0][3];
	m_Matrix.c4x4[1][0] =  src.m_Matrix.c4x4[1][0];
	m_Matrix.c4x4[1][1] =  src.m_Matrix.c4x4[1][1];
	m_Matrix.c4x4[1][2] = -src.m_Matrix.c4x4[1][2];
	m_Matrix.c4x4[1][3] =  src.m_Matrix.c4x4[1][3];
	m_Matrix.c4x4[2][0] = -src.m_Matrix.c4x4[2][0];
	m_Matrix.c4x4[2][1] = -src.m_Matrix.c4x4[2][1];
	m_Matrix.c4x4[2][2] =  src.m_Matrix.c4x4[2][2];
	m_Matrix.c4x4[2][3] = -src.m_Matrix.c4x4[2][3];
	m_Matrix.c4x4[3][0] =  src.m_Matrix.c4x4[3][0];
	m_Matrix.c4x4[3][1] =  src.m_Matrix.c4x4[3][1];
	m_Matrix.c4x4[3][2] = -src.m_Matrix.c4x4[3][2];
	m_Matrix.c4x4[3][3] =  src.m_Matrix.c4x4[3][3];
}


/////////////////////////////////////////////////////////////////////////////
//
//	LogMatrix()
//
void QzMatrix4x4::LogMatrix(void)
{
	QzMatrix4x4_t &m = m_Matrix;

	UtfFormat fmt;
	fmt.AddFloat(m.c4x4[0][0]);
	fmt.AddFloat(m.c4x4[0][1]);
	fmt.AddFloat(m.c4x4[0][2]);
	fmt.AddFloat(m.c4x4[0][3]);
	LogMessage("Matrix: %1w8.5; %2w8.5; %3w8.5; %4w8.5;", fmt);

	fmt.Reset();
	fmt.AddFloat(m.c4x4[1][0]);
	fmt.AddFloat(m.c4x4[1][1]);
	fmt.AddFloat(m.c4x4[1][2]);
	fmt.AddFloat(m.c4x4[1][3]);
	LogMessage("        %1w8.5; %2w8.5; %3w8.5; %4w8.5;", fmt);

	fmt.Reset();
	fmt.AddFloat(m.c4x4[2][0]);
	fmt.AddFloat(m.c4x4[2][1]);
	fmt.AddFloat(m.c4x4[2][2]);
	fmt.AddFloat(m.c4x4[2][3]);
	LogMessage("        %1w8.5; %2w8.5; %3w8.5; %4w8.5;", fmt);

	fmt.Reset();
	fmt.AddFloat(m.c4x4[3][0]);
	fmt.AddFloat(m.c4x4[3][1]);
	fmt.AddFloat(m.c4x4[3][2]);
	fmt.AddFloat(m.c4x4[3][3]);
	LogMessage("        %1w8.5; %2w8.5; %3w8.5; %4w8.5;", fmt);
}


/////////////////////////////////////////////////////////////////////////////
//
//	PrintMatrix()
//
void QzMatrix4x4::PrintMatrix(void)
{
	QzMatrix4x4_t &m = m_Matrix;

	printf("Matrix: %8.5f %8.5f %8.5f %8.5f\n", m.c4x4[0][0], m.c4x4[0][1], m.c4x4[0][2], m.c4x4[0][3]);
	printf("        %8.5f %8.5f %8.5f %8.5f\n", m.c4x4[1][0], m.c4x4[1][1], m.c4x4[1][2], m.c4x4[1][3]);
	printf("        %8.5f %8.5f %8.5f %8.5f\n", m.c4x4[2][0], m.c4x4[2][1], m.c4x4[2][2], m.c4x4[2][3]);
	printf("        %8.5f %8.5f %8.5f %8.5f\n", m.c4x4[3][0], m.c4x4[3][1], m.c4x4[3][2], m.c4x4[3][3]);
	printf("\n");
}

