/////////////////////////////////////////////////////////////////////////////
//
//	File: QzMainMac.h
//
//	$Header: /Projects/Qz/QzMainMac.h  2  2009/9/14 5:44:08p  Lee $
//
//
//	This code uses Carbon to interface with the OS, OpenGL for all rendering,
//	and is designed to support a cross-platform app that can compile and
//	build on both Windows and OSX.
//
//	This file contains two classes: a main app class, and a window class.
//	This is necessary with Carbon, since an app is assumed to be created
//	and kept alive forever, periodically creating and destroying windows as
//	needed, and potentially having multiple windows open at the same time.
//
//	This is slightly at odds with Windows, where an app often only handles
//	one file or window -- supporting multiple files/windows is possible, but
//	the app must be designed for it.
//
//	Since this code is used as a front-end for creating toy games and
//	graphical utilities, and be cross-platform for both Windows and MacOS,
//	it is not designed to support multiple windows.  Although a separate app
//	and window class exist, the app class only allows a single window to
//	exist at a single time.  Supporting multiple OpenGL rendering windows is
//	problematic on Windows, so no attempt is made to support that on the Mac.
//
//	For a detailed write-up of what this code does, consult
//		http://www.teachsolaisgames.com/articles/crossplat4.html
//
/////////////////////////////////////////////////////////////////////////////


#pragma once


class QzLogger;
class QzManager;


class QzMainWin
{
private:
	WindowRef         m_hWindow;
	EventHandlerUPP   m_EventHandlerUPP;
	EventHandlerRef   m_EventHandlerRef;
	EventLoopTimerUPP m_TimerUPP;
	EventLoopTimerRef m_hTimer;
	HIRect            m_OutBounds;				// in window coordinate space
	HIRect            m_WindowPos;				// in display coordinate space
	AGLContext        m_AglContext;

	int m_WindowWidth;
	int m_WindowHeight;
	int m_MousePosX;
	int m_MousePosY;
	U32 m_LastDrawTime;

	QzManager*  m_pManager;

public:
	QzMainWin(void);
	~QzMainWin(void);

	OSStatus LaunchWindow(void);
	void     CloseWindow(void);
	void     CheckForWindowInvalidation(void);

	// Window-level event handlers:
	OSStatus HandleEvent(EventHandlerCallRef hCaller, EventRef hEvent);
	OSStatus HandleEventClassCommand(EventRef hEvent);
	OSStatus HandleEventCommand(EventRef hEvent);
	OSStatus HandleEventClassWindow(EventRef hEvent);
	OSStatus HandleEventClassMouse(EventRef hEvent);
	OSStatus HandleEventClassKeyboard(EventRef hEvent);
	OSStatus HandleEventClassText(EventRef hEvent);

	void HandleTimer(EventLoopTimerRef hTimer);

	void UpdateKeyState(EventRef hEvent, bool isDown);
	void UpdateTextInput(EventRef hEvent);
	void UpdateMouseClick(EventRef hEvent, bool isDown);
	void UpdateMousePosition(EventRef hEvent);
	void UpdateMouseWheel(EventRef hEvent);


	//
	// The rest of these methods are static stub functions used for callbacks
	// from Carbon.  The pContext is the *this pointer, so we'll need to cast
	// it, then call into the actual handler method defined within this class.
	//

	static OSStatus StubWindowEvent(EventHandlerCallRef hCaller, EventRef hEvent, void *pContext)
	{
		return reinterpret_cast<QzMainWin*>(pContext)->HandleEvent(hCaller, hEvent);
	}

	static pascal void StubTimer(EventLoopTimerRef hTimer, void *pContext)
	{
		reinterpret_cast<QzMainWin*>(pContext)->HandleTimer(hTimer);
	}
};


class QzMainApp
{
private:
	QzLogger*   m_pLogger;
	QzMainWin*	m_pWindow;

public:
	QzMainApp(void);
	~QzMainApp(void);

	OSStatus LaunchApp(void);

	// App-level event handlers:
	OSStatus HandleEvent(EventHandlerCallRef hCaller, EventRef hEvent);
	OSStatus HandleEventClassCommand(EventRef hEvent);
	OSStatus HandleEventCommand(EventRef hEvent);
	OSStatus HandleEventClassApp(EventRef hEvent);

	bool EngineInit(Utf08_t cmdLine[]);
	void EngineUninit(void);


	//
	// The rest of these methods are static stub functions used for callbacks
	// from Carbon.  The pContext is the *this pointer, so we'll need to cast
	// it, then call into the actual handler method defined within this class.
	//

	static OSStatus StubAppEvent(EventHandlerCallRef hCaller, EventRef hEvent, void *pContext)
	{
		return reinterpret_cast<QzMainApp*>(pContext)->HandleEvent(hCaller, hEvent);
	}
};



