/////////////////////////////////////////////////////////////////////////////
//
//	File: QzMainMac.cpp
//
//	$Header: /Projects/Qz/QzMainMac.cpp  3  2009/9/14 5:44:08p  Lee $
//
//
//	This code uses Carbon to interface with the OS, OpenGL for all rendering,
//	and is designed to support a cross-platform app that can compile and
//	build on both Windows and OSX.
//
//	This file contains two classes: a main app class, and a window class.
//	This is necessary with Carbon, since an app is assumed to be created
//	and kept alive forever, periodically creating and destroying windows as
//	needed, and potentially having multiple windows open at the same time.
//
//	This is slightly at odds with Windows, where an app often only handles
//	one file or window -- supporting multiple files/windows is possible, but
//	the app must be designed for it.
//
//	Since this code is used as a front-end for creating toy games and
//	graphical utilities, and be cross-platform for both Windows and MacOS,
//	it is not designed to support multiple windows.  Although a separate app
//	and window class exist, the app class only allows a single window to
//	exist at a single time.  Supporting multiple OpenGL rendering windows is
//	problematic on Windows, so no attempt is made to support that on the Mac.
//
//	For a detailed write-up of what this code does, consult
//		http://www.teachsolaisgames.com/articles/crossplat4.html
//
/////////////////////////////////////////////////////////////////////////////


#include <Carbon/Carbon.h>
#include <OpenGL/OpenGL.h>
#include <AGL/agl.h>
#include "QzCommon.h"
#include "QzLogger.h"
#include "QzMainMac.h"
#include "QzManager.h"
#include "UtfData.h"
#include <libgen.h>


#define c_TargetFPS		30.0f


// List of app-level events that are handled by QzMainApp.
static const EventTypeSpec g_AppEvents[] =
{
	{ kEventClassCommand,     kEventCommandProcess },
	{ kEventClassApplication, kEventAppActivated },
	{ kEventClassApplication, kEventAppDeactivated },
	{ kEventClassApplication, kEventAppHidden },
	{ kEventClassApplication, kEventAppShown },
};


// List of window-level events that are handled by QzMainWin.
static const EventTypeSpec g_WindowEvents[] =
{
	{ kEventClassCommand,   kEventCommandProcess },
	{ kEventClassWindow,    kEventWindowActivated },
	{ kEventClassWindow,    kEventWindowDeactivated },
	{ kEventClassWindow,    kEventWindowBoundsChanging },
	{ kEventClassWindow,    kEventWindowBoundsChanged },
	{ kEventClassWindow,    kEventWindowClose },
	{ kEventClassWindow,    kEventWindowClosed },
	{ kEventClassWindow,    kEventWindowDragStarted },
	{ kEventClassWindow,    kEventWindowDragCompleted },
	{ kEventClassWindow,    kEventWindowDrawContent },
	{ kEventClassWindow,    kEventWindowGetMinimumSize },
	{ kEventClassWindow,    kEventWindowResizeCompleted },
	{ kEventClassWindow,    kEventWindowResizeStarted },
	{ kEventClassMouse,     kEventMouseButtonPrimary },
	{ kEventClassMouse,     kEventMouseButtonSecondary },
	{ kEventClassMouse,     kEventMouseButtonTertiary },
	{ kEventClassMouse,     kEventMouseDragged },
	{ kEventClassMouse,     kEventMouseMoved },
	{ kEventClassMouse,     kEventMouseWheelMoved },
	{ kEventClassKeyboard,  kEventRawKeyDown },
	{ kEventClassKeyboard,  kEventRawKeyUp },
	{ kEventClassTextInput, kEventTextInputUnicodeForKeyEvent },
};


/////////////////////////////////////////////////////////////////////////////
//
//	QzAssertHandler()
//
//	Custom assertion handler.  Deals with assertions without throwing up a
//	dialog.
//
void QzAssertHandler(char message[], U32 lineNum, char file[])
{
	UtfFormat fmt;
	fmt.AddInt(lineNum);
	fmt.AddString(basename(file));
	fmt.AddString(message);

	// Log the message.
	LogMessage("Assert: line %1;, file %2;\n\n%3;\n", fmt);

	// Depending on preferences, this could terminate the app, break into
	// the debugger, or keep running and hope for the best.
}


/////////////////////////////////////////////////////////////////////////////
//
//	ParseCommandParam()
//
//	This will be called once for each argv[] string given to main().  If the
//	app were expecting any info on the command line, this is where we would
//	parse it.
//
void ParseCommandParam(Utf08_t cmd[])
{
	if ((NULL == cmd) || ('\0' == cmd[0])) {
		return;
	}

//	if (0 == UtfCompareNocase(cmd, reinterpret_cast<const Utf08_t*>("-param"))) {
//	}
//	if (UtfPrefixNocase(reinterpret_cast<const Utf08_t*>("-param="), cmd)) {
//	}
}



/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzMainWin::QzMainWin(void)
	:	m_hWindow(NULL),
		m_EventHandlerUPP(NULL),
		m_EventHandlerRef(NULL),
		m_TimerUPP(NULL),
		m_hTimer(NULL),
		m_AglContext(NULL),
		m_WindowWidth(800),
		m_WindowHeight(600),
		m_MousePosX(100),
		m_MousePosY(50),
		m_pManager(NULL)
{
	m_LastDrawTime = QzGetMilliseconds();
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzMainWin::~QzMainWin(void)
{
	CloseWindow();

	if (NULL != m_hTimer) {
		RemoveEventLoopTimer(m_hTimer);
	}

	if (NULL != m_EventHandlerRef) {
		RemoveEventHandler(m_EventHandlerRef);
	}

	if (NULL != m_EventHandlerUPP) {
		DisposeEventHandlerUPP(m_EventHandlerUPP);
	}

	SafeDelete(m_pManager);

	if (NULL != m_AglContext) {
		aglDestroyContext(m_AglContext);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	LaunchWindow()
//
//	This creates a new window, initializes an OpenGL context to render to the
//	window, and creates the QzManager that is responsible for handling all of
//	the rendering.
//
OSStatus QzMainWin::LaunchWindow(void)
{
	// Protect against reentrancy.  Do not attempt to create a new window if
	// it is already managing one.
	if (NULL != m_pManager) {
		return -1;
	}

	// Caution when using the Rect struct: the components are arranged as
	// top/left/bottom/right, which is different from Windows.  Trying to
	// automatically init it with values between { and } will result in the
	// horizontal and vertical values being swapped if you're not paying
	// attention to the convention.
    Rect windowRect;
	windowRect.left   = 0;
	windowRect.right  = m_WindowWidth;
	windowRect.top    = 0;
	windowRect.bottom = m_WindowHeight;

    GLint attribs[] =
    {
        AGL_DOUBLEBUFFER,
        AGL_ACCELERATED,
        AGL_NO_RECOVERY,
        AGL_RGBA,
		AGL_DEPTH_SIZE, 32,
        AGL_NONE
    };

	// Create the OpenGL render context that will be used for drawing within
	// this window.
	AGLPixelFormat pixelFormat = aglChoosePixelFormat(NULL, 1, attribs);
	m_AglContext = aglCreateContext(pixelFormat, NULL);
	aglDestroyPixelFormat(pixelFormat);
	aglSetCurrentContext(m_AglContext);

	// These flags control the look of the window, its border, shadows behind
	// the window, etc.
	WindowAttributes attribFlags = kWindowStandardDocumentAttributes
								 | kWindowStandardHandlerAttribute
								 | kWindowResizableAttribute
								 | kWindowLiveResizeAttribute
								 | kWindowNoShadowAttribute
								 | kWindowCloseBoxAttribute;

    // Create the window that will be used for drawing.  It is possible
	// to create this window using CreateWindowFromNib(), but that
	// requires a window to be defined in the NIB, and that window must
	// be flagged to have the standard handlers loaded.  If the standard
	// handlers are not loaded, not all events will be processed, even
	// if window-specific handlers are loaded.  An example of this is
	// mouse events: these events will never be received if the standard
	// handlers are not loaded.  Either create a window that explicitly
	// has kWindowStandardHandlerAttribute defined, or make certain that
	// he window definition in the NIB has this flag set.
	//
	CreateNewWindow(kDocumentWindowClass, attribFlags, &windowRect, &m_hWindow);

	m_EventHandlerUPP = NewEventHandlerUPP(StubWindowEvent);

	// Install the event handler for the window.  This will receive all of
	// the window-level events, such as cut/copy/paste requests along with
	// mouse and keyboard input.
	InstallEventHandler(GetWindowEventTarget(m_hWindow), m_EventHandlerUPP, GetEventTypeCount(g_WindowEvents), g_WindowEvents, this, &m_EventHandlerRef);

	// Define a timer that will wake up the app every few milliseconds
	// to prompt the window to render the next frame with OpenGL.
	m_TimerUPP = NewEventLoopTimerUPP(StubTimer);
	EventTimerInterval delay = kEventDurationSecond / c_TargetFPS;
	InstallEventLoopTimer(GetMainEventLoop(), delay, delay, m_TimerUPP, this, &m_hTimer);

    // Position the window in the middle of the screen.
	RepositionWindow(m_hWindow, NULL, kWindowCenterOnMainScreen);

    // Windows are created in the hidden state, so we need to explicitly make
	// them visible.
	ShowWindow(m_hWindow);

	// Assign the OpenGL context to this window so we will be able to draw
	// to it.
	aglSetWindowRef(m_AglContext, m_hWindow);
	aglUpdateContext(m_AglContext);

	// Request the bounds of the current window.  This is important since
	// the window that gets created will be larger than requested.  Extra
	// space is reserved at the top of the window for the menu bar (22 lines
	// on the system used for testing).  Getting the bounds will tell us
	// what the actual window size is, and the relative position of the
	// upper left corner, which needs to be subtracted from mouse coords
	// to map the mouse to the correct position within the window.
	HIWindowGetBounds(m_hWindow, kWindowContentRgn, kHICoordSpaceScreenPixel, &m_WindowPos);
	HIWindowGetBounds(m_hWindow, kWindowContentRgn, kHICoordSpaceWindow, &m_OutBounds);
	m_WindowWidth  = QzFloatToInt(m_OutBounds.size.width);
	m_WindowHeight = QzFloatToInt(m_OutBounds.size.height);
	UtfFormat fmt;
	fmt.AddInt(QzFloatToInt(m_OutBounds.origin.x));
	fmt.AddInt(QzFloatToInt(m_OutBounds.origin.y));
	fmt.AddInt(m_WindowWidth);
	fmt.AddInt(m_WindowHeight);
	LogMessage("initial bounds: x=%1;, y=%2;, w=%3;, h=%4;", fmt);

	m_pManager = new QzManager;

	if (false == m_pManager->Initialize()) {
		return -1;
	}

	// Request the window title from the manager.  This will be a UTF-8
	// string, so we can give it directly to MacOS.  In order to provide the
	// string to the OS, we have to encapsulate it in a CFString object.
	Utf08_t appTitle[64];
	m_pManager->GetAppTitle(appTitle, ArraySize(appTitle));
	Utf16_t title16[64];
	U32 charCount = UtfConvert08to16(title16, ArraySize(title16), appTitle);
	CFStringRef titleRef = CFStringCreateWithCharacters(NULL, title16, charCount);
	SetWindowTitleWithCFString(m_hWindow, titleRef);

	// Now that everything is set up with the window (mostly for creating
	// the AGL context required for rendering), we can now create the OpenGL
	// renderer that will be used.
	if (false == m_pManager->CreateRenderer()) {
		return -1;
	}

	m_pManager->SetResolution(m_WindowWidth, m_WindowHeight);

	return noErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	CloseWindow()
//
void QzMainWin::CloseWindow(void)
{
	SafeDelete(m_pManager);
}


/////////////////////////////////////////////////////////////////////////////
//
//	CheckForWindowInvalidation()
//
//	This will be called periodically to determine whether the window needs
//	to be redrawn.  Assuming that the window is active, this will trigger a
//	window redraw if the window is dirty (its internal state has changed),
//	or if more than a certain period of time has passed (as determined by
//	GetMaxFrameDelay()).
//
void QzMainWin::CheckForWindowInvalidation(void)
{
	// Only force the window to redraw itself if it is visible.
	// This prevents minimized windows from wasting system resources.
	if (IsWindowActive(m_hWindow)) {
		U32 timeStamp = QzGetMilliseconds();

		if (m_pManager->IsWindowDirty() ||
			((timeStamp - m_LastDrawTime) > m_pManager->GetMaxFrameDelay()))
		{
			m_pManager->ClearDirtyFlag();

			Rect rect;

			// Invalidate the window using window coordinates.
			rect.left   = 0;
			rect.right  = m_OutBounds.origin.x + m_OutBounds.size.width;
			rect.top    = 0;
			rect.bottom = m_OutBounds.origin.y + m_OutBounds.size.height;

			InvalWindowRect(m_hWindow, &rect);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEvent()
//
OSStatus QzMainWin::HandleEvent(EventHandlerCallRef hCaller, EventRef hEvent)
{
	switch (GetEventClass(hEvent)) {
		case kEventClassCommand:
			return HandleEventClassCommand(hEvent);

		case kEventClassWindow:
			return HandleEventClassWindow(hEvent);

		case kEventClassMouse:
			return HandleEventClassMouse(hEvent);

		case kEventClassKeyboard:
			return HandleEventClassKeyboard(hEvent);

		case kEventClassTextInput:
			return HandleEventClassText(hEvent);
	}

	CheckForWindowInvalidation();


	// Always return eventNotHandledErr if we don't handle the event.
	// Assuming that the events were registered correctly, the only events
	// we should ever see are the ones we requested, so we should not ever
	// get to this point.
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassCommand()
//
OSStatus QzMainWin::HandleEventClassCommand(EventRef hEvent)
{
	switch (GetEventKind(hEvent)) {
		case kEventCommandProcess:
			return HandleEventCommand(hEvent);
	}

 	// Always return eventNotHandledErr if we don't handle the event.
	// Assuming that the events were registered correctly, the only events
	// we should ever see are the ones we requested, so we should not ever
	// get to this point.
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventCommand()
//
OSStatus QzMainWin::HandleEventCommand(EventRef hEvent)
{
	HICommandExtended cmd;
	OSStatus result = GetEventParameter(hEvent, kEventParamDirectObject, typeHICommand, NULL, sizeof(cmd), NULL, &cmd);

	if (0 != result) {
		return result;
	}

	// This is a sampling of the types of commands that will be directed
	// to this method.  Depending on the nature of the app, some of these
	// commands may be useful.
	switch (cmd.commandID) {
		case kHICommandClear:
			printf("clear\n");
			return noErr;

		case kHICommandCopy:
			printf("copy\n");
			return noErr;

		case kHICommandCut:
			printf("cut\n");
			return noErr;

		case kHICommandPaste:
			printf("paste\n");
			return noErr;

		case kHICommandRedo:
			printf("redo\n");
			return noErr;

		case kHICommandSelectAll:
			printf("select all\n");
			return noErr;

		case kHICommandUndo:
			printf("undo\n");
			return noErr;
	}

	// Always return eventNotHandledErr if we don't handle the event.
	//
	// Note that here, since we've requested commands (which is a catch-all
	// category), we will receive every command through this function, even
	// the ones we don't need.  Most of those commands should be handled by
	// the standard event handlerd defined in Carbon.  In order for those
	// commands to be passed on to the standard handler, we must return
	// eventNotHandledErr.
	//
	// Also note that you may want to do something special for a command,
	// and still return eventNotHandledErr so that the standard handler
	// will also apply its default handling of the command.
	//
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassWindow()
//
//	This handler will receive all of the window-level events, generally
//	pertaining to moving, resizing, or hiding the window.
//
OSStatus QzMainWin::HandleEventClassWindow(EventRef hEvent)
{
	WindowRef hWindow;
	OSStatus result = GetEventParameter(hEvent, kEventParamDirectObject, typeWindowRef, NULL, sizeof(hWindow), NULL, &hWindow);

	if (0 != result) {
		return result;
	}

	// There is only one possible window that could exist, and that is the
	// only window handle we should expect to see here.
	if (hWindow != m_hWindow) {
		return eventNotHandledErr;
	}

	UInt32 eventID = GetEventKind(hEvent);

	switch (eventID) {
		case kEventWindowActivated:
			printf("activated\n");

			// Return eventNotHandledErr so the default handler will update
			// the titlebar to reflect the current state.
			return eventNotHandledErr;

		case kEventWindowDeactivated:
			printf("deactivated\n");

			// Return eventNotHandledErr so the default handler will update
			// the titlebar to reflect the current state.
			return eventNotHandledErr;

		// This indicates that the window is going to be closed.  Hook this
		// event if the app needs to interrupt the close operation (such as
		// for displaying a "Save Changes?" dialog).
		case kEventWindowClose:
			break;

		// This event indicates that the window has been closed.
		case kEventWindowClosed:
			// This is completely non-standard:  When the window is closed,
			// this app will terminate itself.  Normally, a MacOS app should
			// stay running at all times, even when all windows have been
			// closed.
			//
			// When writing a cross-platform app, the Windows version of the
			// app will need to terminate when its main window is closed.
			// This may result in start-up/shut-down logic that requires the
			// whole app (regardless of platform) to terminate when the main
			// window is closed.
			//
			QuitApplicationEventLoop();
			break;

		// What is the recommended size for a window?
		case kEventWindowGetIdealSize:
			break;

		// OSX wants to know what minimum size is required for the window.
		// This should be used to enforce a minimum size if the user is
		// trying to shrink the window.
		case kEventWindowGetMinimumSize:
			{
				HIPoint size;
				size.x = 800;
				size.y = 600;
				SetEventParameter(hEvent, kEventParamDimensions, typeQDPoint, sizeof(size), &size);
			}
			break;

		// The user has start dragging the window around on the screen.  Depending
		// how the window is being used, you could use this event as a signal to
		// stop rendering until the drag is completed.
		//
		// While the drag is occurring, the window will receive pairs of
		// bounds-changing and bounds-changed messages.
		case kEventWindowDragStarted:
			printf("drag started\n");
			break;

		case kEventWindowDragCompleted:
			printf("drag completed\n");
			break;

		case kEventWindowResizeStarted:
			printf("resize started\n");
			break;

		case kEventWindowResizeCompleted:
			printf("resize completed\n");
			break;

		// Window is being resized or moved.  This event will be followed
		// by a bounds-changed event.  While the drag is occurring, a
		// long stream of changing/changed event pairs will arrive.
		// drag-start and drag-stop events.
		case kEventWindowBoundsChanging:
			printf("bounds changing\n");
			break;

		// Window has finished being moved or resized (at least until
		// the next mouse update, when we'll likely receive another
		// changing/changed event pair).
		case kEventWindowBoundsChanged:
			printf("bounds changed\n");

			// Retrieve the position in display coordinates, which is
			// needed for mouse warping.
			HIWindowGetBounds(m_hWindow, kWindowContentRgn, kHICoordSpaceScreenPixel, &m_WindowPos);

			// Then get the coordinates in window space, which is used
			// by most of the message events.
			HIWindowGetBounds(m_hWindow, kWindowContentRgn, kHICoordSpaceWindow, &m_OutBounds);
			m_WindowWidth  = int(m_OutBounds.size.width + 0.5f);
			m_WindowHeight = int(m_OutBounds.size.height + 0.5f);
			aglUpdateContext(m_AglContext);
			if (NULL != m_pManager) {
				m_pManager->SetResolution(m_WindowWidth, m_WindowHeight);
			}
			return noErr;

		// This event indicates that the window needs to redraw itself,
		// either because of a window change (window was resized, restored,
		// or revealed), or because the contents of the window was
		// invalidated by InvalWindowRect().
		case kEventWindowDrawContent:
			if (IsWindowActive(m_hWindow)) {
				m_LastDrawTime = QzGetMilliseconds();
				if (m_pManager->Render()) {
				    aglSwapBuffers(m_AglContext);
				}
			}
			return noErr;
	}

	// Always return eventNotHandledErr if we don't handle the event.
	// Assuming that the events were registered correctly, the only events
	// we should ever see are the ones we requested, so we should not ever
	// get to this point.
 	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassMouse()
//
OSStatus QzMainWin::HandleEventClassMouse(EventRef hEvent)
{
	switch (GetEventKind(hEvent)) {
		case kEventMouseDown:
			UpdateMousePosition(hEvent);
			UpdateMouseClick(hEvent, true);
			break;

		case kEventMouseUp:
			UpdateMousePosition(hEvent);
			UpdateMouseClick(hEvent, false);
			break;

		// Dragged events are really Moved events.  If any mouse button
		// is being held down while the mouse moves, the window gets a
		// Dragged event.  If no mouse buttons are held down, then a
		// normal Moved event is received instead.
		case kEventMouseDragged:
			UpdateMousePosition(hEvent);
			break;

		case kEventMouseMoved:
			UpdateMousePosition(hEvent);
			break;

		case kEventMouseWheelMoved:
			UpdateMousePosition(hEvent);
			UpdateMouseWheel(hEvent);
			break;
	}

	// Always report that we did not handle the mouse, since we're only
	// interested in button state.  We get mouse events for clicks on the
	// window border (if there is one in the current window mode), so we
	// need to return this value so close events are handled correctly.
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassKeyboard()
//
//	Keyboard events are used to detect when a key is pressed or released,
//	so we can track the state of all keys.
//
OSStatus QzMainWin::HandleEventClassKeyboard(EventRef hEvent)
{
	switch (GetEventKind(hEvent)) {
		case kEventRawKeyDown:
			UpdateKeyState(hEvent, true);
			break;

		case kEventRawKeyUp:
			UpdateKeyState(hEvent, false);
			break;
	}

	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassText()
//
//	Text events indicate the user it typing.  These arrive along with key
//	events.  The difference is that key events provide magic key codes that
//	are different with every Mac keyboard (because consistency is the bane
//	of Apple's existence), whereas text events have been translated to
//	UTF character codes.
//
OSStatus QzMainWin::HandleEventClassText(EventRef hEvent)
{
	switch (GetEventKind(hEvent)) {
		case kEventTextInputUnicodeForKeyEvent:
			UpdateTextInput(hEvent);
			return noErr;
	}

	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleTimer()
//
//	This timer function will be called every X milliseconds.  All it needs
//	to do is invalidate the window.  At some point after this, the window
//	will receive a kEventWindowDrawContent event that will cause the window
//	to redraw.
//
void QzMainWin::HandleTimer(EventLoopTimerRef hTimer)
{
	CheckForWindowInvalidation();
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdateKeyState()
//
//	Keyboard input is highly problematic on a Mac.  Virtual key codes cannot
//	be trusted, since the exact values sometimes change across different
//	versions of OSX, and even OS patches for the same version.  Virtual codes
//	also differ with localization and keyboards (AZERTY vs QWERTY).  For some
//	keyboards, different keys may report the same code value.
//
//	Using char codes is more reliable, but has its own set of problems.  The
//	key-down and key-up events report the char code based on the current
//	modifier keys.  If the user presses 'a', presses shift, then releases 'a',
//	key-down reports 'a', but key-up reports 'A'.  While the result of the
//	shift key is predictable and could be compensated for, the control, alt,
//	and command keys map the char value to unrelated values, making it very
//	difficult to reliably detect when specific keys have been pressed or
//	released.
//
//	There have been several hacks over different versions of OSX to figure
//	out exactly what key has been pressed, and all have been deprecated.
//	Quite frankly, I gave up on figuring it out.  This is a hack that works
//	on my system.  Use with caution, and don't expect it to work right for
//	other keyboard layouts or future changes to OSX.
//
//	When writing a standard app, this is probably never going to be an issue.
//	But for any app that makes specialized use of the keyboard (such as
//	games), the need to raw input is important, but the reliability of raw
//	input is virtually nil due to inconsistencies between keyboards.
//
void QzMainWin::UpdateKeyState(EventRef hEvent, bool isDown)
{
	OSStatus result;

	UInt32 modifiers = 0;
	result = GetEventParameter(hEvent, kEventParamKeyModifiers, typeUInt32, NULL, sizeof(modifiers), NULL, &modifiers);

	if (0 != result) {
		printf("Get failed\n");
	}

	U32 keyFlags = 0;
	if (cmdKey & modifiers)     { keyFlags |= QzKeyMask_Control; }
	if (shiftKey & modifiers)   { keyFlags |= QzKeyMask_Shift; }
	if (alphaLock & modifiers)  { keyFlags |= QzKeyMask_CapsLock; }
	if (optionKey & modifiers)  { keyFlags |= QzKeyMask_Alt; }
	if (controlKey & modifiers) { printf(" controlKey"); }
	if (kEventKeyModifierNumLockMask & modifiers) { keyFlags |= QzKeyMask_NumLock; }

	char symbol;
	result = GetEventParameter(hEvent, kEventParamKeyMacCharCodes, typeChar, NULL, sizeof(symbol), NULL, &symbol);

	if (0 != result) {
//		return;
	}

	UInt32 keyCode;
	result = GetEventParameter(hEvent, kEventParamKeyCode, typeUInt32, NULL, sizeof(keyCode), NULL, &keyCode);

	if (0 != result) {
//		return;
	}

	if (('a' <= symbol) && (symbol <= 'z')) {
		symbol -= 'a' - 'A';
	}

//	printf("key: %d %02X %c %s %05X\n", keyCode, symbol, symbol, isDown ? "down" : "up", modifiers);

	// Alert the manager to the key event, so it can keep track of the state
	// of all keys.
	if (NULL != m_pManager) {
		m_pManager->KeyPress(keyFlags, symbol, isDown, false);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdateTextInput()
//
//	This will handle receiving text from keyboard events.  This text will be
//	in UTF-8 format, which is a multi-byte representation of Unicode symbols,
//	comprised of 21-bit values in the range 0x0000 to 0x10FFFF.  Each 21-bit
//	value is encoded as a sequence of one byte (for 7-bit values), two bytes
//	(11-bit values), three bytes (16-bit values), or four bytes (21 bits).
//
void QzMainWin::UpdateTextInput(EventRef hEvent)
{
	UTF8Char  sequence[8];
	ByteCount actualSize = 8;

	OSStatus result = GetEventParameter(hEvent, kEventParamTextInputSendText, typeUTF8Text, NULL, actualSize, &actualSize, &sequence);

	if (0 != result) {
		return;
	}

	// Terminate the string.  We may want to print it for debugging,
	// and it needs to be composed and converted to UTF-32.
	sequence[actualSize] = '\0';
//	printf("UTF-8 \"%s\" (%d bytes)\n", sequence, actualSize);

	Utf08_t composed[8];

	// Compose the sequence to make certain it is a known symbol.  If the
	// code cannot figure out what character this is, it will ignore it (this
	// UTF library is only aware of Latin characters, so it will reject CJKV
	// and other non-Latin characters).  Composition will also merge
	// normalized symbol sequences into a single symbol, which is needed
	// to represent the symbol as a single UTF-32 value.
	if (UtfCompose08to08(composed, ArraySize(composed), sequence) > 0) {
		// This only needs to hold two chars: the composed 32-bit char,
		// and space for a 32-bit '\0' to terminate the one-symbol string.
		Utf32_t xform[2];

		// Convert that to UTF-32 so we can process it internally.  A single
		// 32-bit value is much easier to pass around than a variable number
		// of bytes.
		if (UtfConvert08to32(xform, ArraySize(xform), composed)) {
			if (('\0' != xform[0]) && (NULL != m_pManager)) {
				m_pManager->KeyChar(xform[0]);
			}
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdateMouseClick()
//
void QzMainWin::UpdateMouseClick(EventRef hEvent, bool isDown)
{
	// Which button was pressed?
	//   1 = left
	//   2 = right
	//   3 = middle
	EventMouseButton buttonNum = 1;
	OSStatus result = GetEventParameter(hEvent, kEventParamMouseButton, typeMouseButton, NULL, sizeof(buttonNum), NULL, &buttonNum);

	if (0 != result) {
		return;
	}

	// What is the state of the modifier keys (shift, control, command, etc.)?
	UInt32 modifierKeys = 0;
	result = GetEventParameter(hEvent, kEventParamKeyModifiers, typeUInt32, NULL, sizeof(modifierKeys), NULL, &modifierKeys);

	if (0 != result) {
		return;
	}

	U32 mouseFlags = 0;

	// Note: By convention, Apple's "Command" key is treated the same as the
	// Windows "Ctrl" key.
	if (cmdKey & modifierKeys)   { mouseFlags |= QzMouseFlag_Control; }
	if (shiftKey & modifierKeys) { mouseFlags |= QzMouseFlag_Shift; }

//	The following modifier keys are ignored for mouse clicks, since these
//	are not normally used as mouse-click modifiers on Windows.
//
//	if (alphaLock & modifierKeys)  { printf(" capsLock"); }
//	if (optionKey & modifierKeys)  { printf(" optionKey"); }
//	if (controlKey & modifierKeys) { printf(" controlKey"); }
//	if (kEventKeyModifierNumLockMask & modifierKeys) { printf(" kEventKeyModifierNumLockMask"); }

	QzMouseButton_t buttonID = QzMouseButton_Left;
	if (2 == buttonNum) {
		buttonID = QzMouseButton_Right;
	}
	else if (3 == buttonNum) {
		buttonID = QzMouseButton_Middle;
	}

	QzMouseClick_t click = isDown ? QzMouseClick_Down : QzMouseClick_Up;

	if (NULL != m_pManager) {
		m_pManager->MouseClick(buttonID, click, mouseFlags, m_MousePosX, m_MousePosY);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdateMousePosition()
//
void QzMainWin::UpdateMousePosition(EventRef hEvent)
{
	// The normal case is when the mouse is not locked to the window.  Here
	// we just need to forward the new window-relative position of the mouse
	// into the manager.
	if (false == QzIsMouseLocked()) {
		// Warning: This struct uses floats.
		HIPoint position;

		OSStatus result = GetEventParameter(hEvent, kEventParamWindowMouseLocation, typeHIPoint, NULL, sizeof(position), NULL, &position);

		if (0 == result) {
			S32 newX = QzFloatToInt(position.x);
			S32 newY = QzFloatToInt(position.y - m_OutBounds.origin.y);

			// Ignore the event it the mouse's position has not changed.
			if ((newX != m_MousePosX) || (newY != m_MousePosY)) {
				m_MousePosX = newX;
				m_MousePosY = newY;

				if (NULL != m_pManager) {
					m_pManager->MouseMove(m_MousePosX, m_MousePosY);
				}
			}
		}
	}
	//
	// Otherwise, the mouse is locked.  In this state, the mouse is hidden
	// and forced to stay within the bounds of the window, and the mouse
	// changes are sent in as delta values instead of absolute coordinates.
	// This mode is useful for spinning 3D objects, or when rendering from
	// a first-person point of view.
	//
	else {
		// Warning: This struct uses floats.
		HIPoint delta;

		OSStatus result = GetEventParameter(hEvent, kEventParamMouseDelta, typeHIPoint, NULL, sizeof(delta), NULL, &delta);

		if (0 == result) {
			// To keep the mouse pointer within the bounds of the window, we
			// warp it back to the previous position.  However, warping is
			// done in display coordinates, not window coordinates, so we
			// have to map the window coordinates of the events into display
			// coordinates before we can call the CG routines.

			CGPoint warp;
			warp.x = m_MousePosX + m_WindowPos.origin.x;
			warp.y = m_MousePosY + m_WindowPos.origin.y;
			CGWarpMouseCursorPosition(warp);

			if (NULL != m_pManager) {
				m_pManager->MouseDelta(QzFloatToInt(delta.x), QzFloatToInt(delta.y));
			}
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	UpdateMouseWheel()
//
void QzMainWin::UpdateMouseWheel(EventRef hEvent)
{
	EventMouseWheelAxis axis;
	OSStatus result = GetEventParameter(hEvent, kEventParamMouseWheelAxis, typeMouseWheelAxis, NULL, sizeof(axis), NULL, &axis);

	if (0 != result) {
		return;
	}

	// The mouse wheel interface supports track-ball logic, so wheel events
	// can occur in both the X and Y axis.  Normal mice only have a Y-axis,
	// so ignore X-axis events.
	if (kEventMouseWheelAxisY == axis) {
		SInt32 wheelDelta;
		result = GetEventParameter(hEvent, kEventParamMouseWheelDelta, typeSInt32, NULL, sizeof(wheelDelta), NULL, &wheelDelta);

		if (0 != result) {
			return;
		}

		if (NULL != m_pManager) {
			// For Microsoftian compatibility, multiply the amount of wheel
			// motion by a magic number.  This allows MouseWheel() to process
			// values in the same numerical range on all platforms.
			m_pManager->MouseWheel(wheelDelta * QzMouseWheel_Delta);
		}
	}
}



/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzMainApp::QzMainApp(void)
	:	m_pLogger(NULL),
		m_pWindow(NULL)
{
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzMainApp::~QzMainWin(void)
{
	SafeDelete(m_pWindow);
}


/////////////////////////////////////////////////////////////////////////////
//
//	LaunchApp()
//
//	Initialize the main app object.  This creates the menu bar and registers
//	the handler that will respond to app-level menu commands.
//
OSStatus QzMainApp::LaunchApp(void)
{
	if (NULL != m_pWindow) {
		return noErr;
	}

	OSStatus result;
	IBNibRef hNIB;

	// Create a NIB reference, which is needed to access the definitions
	// within the NIB file.  Give it the name of the NIB file, without the
	// .nib file extension.  CreateNibReference() will search for this NIB
	// within the application bundle.  Note that the NIB will have .xib as
	// its file extension within the project file.  The actual .nib file is
	// created when the project is built.
	result = CreateNibReference(CFSTR("main"), &hNIB);
	if (noErr != result) {
		return result;
	}

	// Once the NIB reference is created, set the menu bar.  "MenuBar" is the
	// name of the menu bar object, as defined in the NIB.  This name is set
	// in InterfaceBuilder when the nib is created.
	result = SetMenuBarFromNib(hNIB, CFSTR("MenuBar"));

	// Dispose of the NIB when we're done with it to avoid resource leaks.
	// Do this here before error checking to assure that this will always
	// be done, regardless of whether the previous function succeeded.
	DisposeNibReference(hNIB);

	if (noErr != result) {
		return result;
	}

	// Install the handler that will receive app-level events.
	InstallEventHandler(GetApplicationEventTarget(), StubAppEvent, GetEventTypeCount(g_AppEvents), g_AppEvents, this, NULL);

	// Create a new window.  This is not normally necessary, since the user
	// can create a new window using the New command (from the menu, keyboard
	// shortcut, or other mechanisms).  We'll explicitly create a new window
	// here so one is visible as soon as the app starts up.
	m_pWindow = new QzMainWin;
	return m_pWindow->LaunchWindow();
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEvent()
//
//	Top-level handler function for events.  Events are nested as three levels
//	of info: class, event, and command.  (The actual terminology seems to be
//	flexible, or different authors aren't consistent about what they call
//	each of the commands.)  Some apps handle this as three nested switch
//	statements, which makes for hard-to-read code.  This code splits the
//	events into separate functions so they're easier to read.
//
OSStatus QzMainApp::HandleEvent(EventHandlerCallRef hCaller, EventRef hEvent)
{
	switch (GetEventClass(hEvent)) {
		case kEventClassCommand:
			return HandleEventClassCommand(hEvent);

		case kEventClassApplication:
			return HandleEventClassApp(hEvent);
	}

	// Always return eventNotHandledErr if we don't handle the event.
	// Assuming that the events were registered correctly, the only events
	// we should ever see are the ones we requested, so we should not ever
	// get to this point.
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassCommand()
//
OSStatus QzMainApp::HandleEventClassCommand(EventRef hEvent)
{
	switch (GetEventKind(hEvent)) {
		case kEventCommandProcess:
			return HandleEventCommand(hEvent);
	}

	// Always return eventNotHandledErr if we don't handle the event.
	// Assuming that the events were registered correctly, the only events
	// we should ever see are the ones we requested, so we should not ever
	// get to this point.
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventCommand()
//
//	This handles the actual app commands, which includes commands from the
//	main app menu on the finder bar.
//
OSStatus QzMainApp::HandleEventCommand(EventRef hEvent)
{
	HICommandExtended cmd;

	OSStatus result = GetEventParameter(hEvent, kEventParamDirectObject, typeHICommand, NULL, sizeof(cmd), NULL, &cmd);

	if (0 != result) {
		return result;
	}

	switch (cmd.commandID) {
		// Do nothing if we get a request to create a new window.  This app
		// only supports one window, since resources cannot have multiple
		// instances.
		case kHICommandNew:
//			return LaunchWindow();
			return noErr;

		// Ignore open commands.  There are no files to open, and we don't
		// want multiple windows active.
		case kHICommandOpen:
			break;
	}

	// Always return eventNotHandledErr if we don't handle the event.
	//
	// Note that here, since we've requested commands, we will receive all
	// types of app commands to this method.  This makes it very important
	// to return eventNotHandledErr, otherwise the standard handler will not
	// process some of the commands it should.
	//
	// Also note that you may want to do something special for a command,
	// and still return eventNotHandledErr so that the standard handler
	// will also apply its default handling of the command.
	//
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	HandleEventClassCommand()
//
OSStatus QzMainApp::HandleEventClassApp(EventRef hEvent)
{
	UInt32 eventID = GetEventKind(hEvent);

	switch (eventID) {
		// Indicates the app regained input focus from another window, or
		// was restored from being minimized on the dock.
		//
		// Rather than try to track state from these messages, it is safer
		// to call IsWindowActive() to test its state.
		//
		case kEventAppActivated:
			printf("app activated\n");
			break;

		// Received when another app takes over the window focus, but not
		// when the app is minimized to the dock.
		case kEventAppDeactivated:
			printf("app deactivated\n");
			break;

		case kEventAppHidden:
			printf("app hidden\n");
			break;

		case kEventAppShown:
			printf("app shown\n");
			break;
	}

	// Always return eventNotHandledErr if we don't handle the event.
	// Assuming that the events were registered correctly, the only events
	// we should ever see are the ones we requested, so we should not ever
	// get to this point.
	return eventNotHandledErr;
}


/////////////////////////////////////////////////////////////////////////////
//
//	EngineInit()
//
//	This handles all of the initialization that can be done before calling
//	RunApplicationEventLoop().
//
bool QzMainApp::EngineInit(Utf08_t cmdLine[])
{
	m_pLogger = new QzLogger("QzLog");
	g_pLog    = m_pLogger;

	// Only enable file logging if the memory logging option is not available.
	// This typically means the logger utility has not been started, or we're
	// running on a Mac, where it isn't supported.
	if (!m_pLogger->MemoryLoggingAvailable()) {
		m_pLogger->Open(reinterpret_cast<const Utf08_t*>("trace.txt"));
	}

	UtfFormat fmt;

	fmt.Reset();
	fmt.AddString(cmdLine);
	LogMessage("command line: \"%1;\"", fmt);

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	EngineUninit()
//
void QzMainApp::EngineUninit(void)
{
	SafeDelete(m_pWindow);

	g_pLog = NULL;

	SafeDelete(m_pLogger);
}


/////////////////////////////////////////////////////////////////////////////
//
//	main()
//
int main(int argc, char *argv[])
{
	Utf08_t commandLine[1024];

	QzSystemInit();
	UtfInitialize();

	// Since we get the command line piecemail, we need to concatenate all
	// the pieces together to form a full command line string for logging.
	SafeStrCopy(commandLine, reinterpret_cast<Utf08_t*>(argv[0]));

	for (int paramNum = 1; paramNum < argc; ++paramNum) {
		ParseCommandParam(reinterpret_cast<Utf08_t*>(argv[paramNum]));

		SafeStrAppend(commandLine, reinterpret_cast<const Utf08_t*>(" "));
		SafeStrAppend(commandLine, reinterpret_cast<Utf08_t*>(argv[paramNum]));
	}

	QzMainApp *pApp = new QzMainApp;

	if (false == pApp->EngineInit(commandLine)) {
		SafeDelete(pApp);
		return 0;
	}

	if (noErr == pApp->LaunchApp()) {
		// Run the event loop.  This will not return until the app is closed.
		RunApplicationEventLoop();
	}

	pApp->EngineUninit();

	delete pApp;

	QzSystemUninit();

	return 0;
}


