/////////////////////////////////////////////////////////////////////////////
//
//	File: QzKeyMapperWin.cpp
//
//	$Header: /TS/TsGui/QzKeyMapperWin.cpp  23  2009/9/8 12:43:41a  Lee $
//
//
//	This class is used to make system keyboard inputs to a standard set of
//	cross-platform symbols, record key events for later processing, and to
//	keep track of the state of each key, and whether any have been pressed or
//	released since the last update.
//
//	Since data processing occurs between world updates, the user may press
//	and release a key between render passes -- this is especially important
//	if the app loses focus or gets minimized.  Multiple key events can occur
//	before the next frame gets rendered.  This state logic allows the app to
//	test is a key was pressed or released, and whether it is still being held
//	down.
//
//	General usage assumes that ClearEvents() is being called after each frame
//	is rendered (or whenever the app is otherwise done processing the inputs
//	for the current frame).  The Reset() method is intended to clear all of
//	the key state information, which is a good idea any time the app has lost
//	keyboard focus, and therefore lost track of the state for each key.
//
/////////////////////////////////////////////////////////////////////////////


#include <windows.h>
#include "QzCommon.h"
#include "QzKeyMapper.h"


// "Sticky keys" are those annoying pop-ups that occur when you hold some
// keys down for several seconds, or press them repeatedly.  The Windows
// functionality allows these keys to be globally disabled.  The keys should
// be disabled when the app is running (but only if the user wants this to
// happen) but they must always be restored to their previous state when the
// app shuts down, gets minimized, or otherwise loses keyboard focus.
//
struct DevInputPrivate_t
{
	bool		StickyKeysDisabled;
	STICKYKEYS	StickyKeys;
	TOGGLEKEYS	ToggleKeys;
	FILTERKEYS	FilterKeys;
};


// This flag controls whether the Windows keys on the keyboard should be
// ignored when running in full-screen.  This would normally be set from
// an INI file or checkbox in a configuration dialog.  The user would set
// this to allow/prevent the Windows key from being disabled.
//
bool g_AllowIgnoreWindowsKey = true;


// This flag control is used for internal control of the Windows key.  It is
// normally set when switching into game mode, then cleared when returning to
// an interface window.  This allows the Windows key to function from config
// windows, but is ignored when in the game.
//
bool g_IgnoreWindowsKey = false;


// This flag needs to be enabled when the app switches into fullscreen mode.
// It is used to determine whether some input operators are ignored or subverted
// when in fullscreen mode.
//
static bool g_FullScreen = false;


// Handle for the low-level keyboard hook that is used to detect and bypass
// some of the normal Windows functionality (such as the Windows key, the
// print screen key).
//
static HHOOK g_hKeyboardHook = NULL;


// This table maps the Windows virtual key codes to known values
//
// Note that this table mostly applies to English keyboard layouts.  There is
// currently no support for "dead keys" or other language-specific keys.
//
U16 g_KeyMap[256] =
{
	0,					// 0x00
	0,					// 0x01 VK_LBUTTON
	0,					// 0x02 VK_RBUTTON
	0,					// 0x03 VK_CANCEL
	0,					// 0x04 VK_MBUTTON
	0,					// 0x05 VK_XBUTTON1
	0,					// 0x06 VK_XBUTTON2
	0,					// 0x07
	QzKey_Backspace,	// 0x08 VK_BACK
	QzKey_Tab,			// 0x09 VK_TAB
	0,					// 0x0A
	0,					// 0x0B
	QzKey_NumpadCenter,	// 0x0C VK_CLEAR
	QzKey_Enter,		// 0x0D VK_RETURN
	0,					// 0x0E
	0,					// 0x0F
	QzKey_Shift,		// 0x10 VK_SHIFT
	QzKey_Control,		// 0x11 VK_CONTROL
	QzKey_Alt,			// 0x12 VK_MENU
	QzKey_Pause,		// 0x13 VK_PAUSE
	QzKey_CapsLock,		// 0x14 VK_CAPITAL
	0,					// 0x15 VK_KANA/VK_HANGEUL/VK_HANGUL
	0,					// 0x16
	0,					// 0x17 VK_JUNJA
	0,					// 0x18 VK_FINAL
	0,					// 0x19 VK_HANJA/VK_KANJI
	0,					// 0x1A
	QzKey_Escape,		// 0x1B VK_ESCAPE
	0,					// 0x1C VK_CONVERT
	0,					// 0x1D VK_NONCONVERT
	0,					// 0x1E VK_ACCEPT
	0,					// 0x1F VK_MODECHANGE
	QzKey_Space,		// 0x20 VK_SPACE
	QzKey_PageUp,		// 0x21 VK_PRIOR
	QzKey_PageDown,		// 0x22 VK_NEXT
	QzKey_End,			// 0x23 VK_END
	QzKey_Home,			// 0x24 VK_HOME
	QzKey_Left,			// 0x25 VK_LEFT
	QzKey_Up,			// 0x26 VK_UP
	QzKey_Right,		// 0x27 VK_RIGHT
	QzKey_Down,			// 0x28 VK_DOWN
	0,					// 0x29 VK_SELECT
	0,					// 0x2A VK_PRINT
	0,					// 0x2B VK_EXECUTE
	QzKey_PrintScreen,	// 0x2C VK_SNAPSHOT
	QzKey_Insert,		// 0x2D VK_INSERT
	QzKey_Delete,		// 0x2E VK_DELETE
	0,					// 0x2F VK_HELP
	QzKey_0,			// 0x30
	QzKey_1,			// 0x31
	QzKey_2,			// 0x32
	QzKey_3,			// 0x33
	QzKey_4,			// 0x34
	QzKey_5,			// 0x35
	QzKey_6,			// 0x36
	QzKey_7,			// 0x37
	QzKey_8,			// 0x38
	QzKey_9,			// 0x39
	0,					// 0x3A
	0,					// 0x3B
	0,					// 0x3C
	0,					// 0x3D
	0,					// 0x3E
	0,					// 0x3F
	0,					// 0x40
	QzKey_A,			// 0x41
	QzKey_B,			// 0x42
	QzKey_C,			// 0x43
	QzKey_D,			// 0x44
	QzKey_E,			// 0x45
	QzKey_F,			// 0x46
	QzKey_G,			// 0x47
	QzKey_H,			// 0x48
	QzKey_I,			// 0x49
	QzKey_J,			// 0x4A
	QzKey_K,			// 0x4B
	QzKey_L,			// 0x4C
	QzKey_M,			// 0x4D
	QzKey_N,			// 0x4E
	QzKey_O,			// 0x4F
	QzKey_P,			// 0x50
	QzKey_Q,			// 0x51
	QzKey_R,			// 0x52
	QzKey_S,			// 0x53
	QzKey_T,			// 0x54
	QzKey_U,			// 0x55
	QzKey_V,			// 0x56
	QzKey_W,			// 0x57
	QzKey_X,			// 0x58
	QzKey_Y,			// 0x59
	QzKey_Z,			// 0x5A
	0,					// 0x5B VK_LWIN
	0,					// 0x5C VK_RWIN
	0,					// 0x5D VK_APPS			Window's Menu key
	0,					// 0x5E VK_APPS
	0,					// 0x5F VK_SLEEP
	QzKey_Insert,		// 0x60 VK_NUMPAD0		these are the numpad values when NumLock is enabled
	QzKey_End,			// 0x61 VK_NUMPAD1		just ignore numlock and map them to direction keys
	QzKey_Down,			// 0x62 VK_NUMPAD2
	QzKey_PageDown,		// 0x63 VK_NUMPAD3
	QzKey_Left,			// 0x64 VK_NUMPAD4
	QzKey_NumpadCenter,	// 0x65 VK_NUMPAD5
	QzKey_Right,		// 0x66 VK_NUMPAD6
	QzKey_Home,			// 0x67 VK_NUMPAD7
	QzKey_Up,			// 0x68 VK_NUMPAD8
	QzKey_PageUp,		// 0x69 VK_NUMPAD9
	QzKey_Multiply,		// 0x6A VK_MULTIPLY
	QzKey_Plus,			// 0x6B VK_ADD
	0,					// 0x6C VK_SEPARATOR
	QzKey_Minus,		// 0x6D VK_SUBTRACT
	QzKey_Decimal,		// 0x6E VK_DECIMAL
	QzKey_Divide,		// 0x6F VK_DIVIDE
	QzKey_F1,			// 0x70 VK_F1
	QzKey_F2,			// 0x71 VK_F2
	QzKey_F3,			// 0x72 VK_F3
	QzKey_F4,			// 0x73 VK_F4
	QzKey_F5,			// 0x74 VK_F5
	QzKey_F6,			// 0x75 VK_F6
	QzKey_F7,			// 0x76 VK_F7
	QzKey_F8,			// 0x77 VK_F8
	QzKey_F9,			// 0x78 VK_F9
	QzKey_F10,			// 0x79 VK_F10		always has ALT key flag
	QzKey_F11,			// 0x7A VK_F11
	QzKey_F12,			// 0x7B VK_F12
	0,					// 0x7C VK_F13
	0,					// 0x7D VK_F14
	0,					// 0x7E VK_F15
	0,					// 0x7F VK_F16
	0,					// 0x80 VK_F17
	0,					// 0x81 VK_F18
	0,					// 0x82 VK_F19
	0,					// 0x83 VK_F20
	0,					// 0x84 VK_F21
	0,					// 0x85 VK_F22
	0,					// 0x86 VK_F23
	0,					// 0x87 VK_F24
	0,					// 0x88
	0,					// 0x89
	0,					// 0x8A
	0,					// 0x8B
	0,					// 0x8C
	0,					// 0x8D
	0,					// 0x8E
	0,					// 0x8F
	QzKey_NumLock,		// 0x90 VK_NUMLOCK
	QzKey_ScrollLock,	// 0x91 VK_SCROLL
	QzKey_Equal,		// 0x92 VK_OEM_NEC_EQUAL =
	0,					// 0x93
	0,					// 0x94
	0,					// 0x95
	0,					// 0x96
	0,					// 0x97
	0,					// 0x98
	0,					// 0x99
	0,					// 0x9A
	0,					// 0x9B
	0,					// 0x9C
	0,					// 0x9D
	0,					// 0x9E
	0,					// 0x9F
	0,					// 0xA0
	0,					// 0xA1
	0,					// 0xA2
	0,					// 0xA3
	0,					// 0xA4
	0,					// 0xA5
	0,					// 0xA6
	0,					// 0xA7
	0,					// 0xA8
	0,					// 0xA9
	0,					// 0xAA
	0,					// 0xAB
	0,					// 0xAC
	0,					// 0xAD
	0,					// 0xAE
	0,					// 0xAF
	0,					// 0xB0
	0,					// 0xB1
	0,					// 0xB2
	0,					// 0xB3
	0,					// 0xB4
	0,					// 0xB5
	0,					// 0xB6
	0,					// 0xB7
	0,					// 0xB8
	0,					// 0xB9
	QzKey_Semicolon,	// 0xBA VK_OEM_1		;:
	QzKey_Equal,		// 0xBB VK_OEM_PLUS		+=
	QzKey_Comma,		// 0xBC VK_OEM_COMMA	,<
	QzKey_Underscore,	// 0xBD VK_OEM_MINUS	-_
	QzKey_Period,		// 0xBE VK_OEM_PERIOD	.>
	QzKey_Slash,		// 0xBF VK_OEM_2		/?
	QzKey_Tilde,		// 0xC0 VK_OEM_3		`~
	0,					// 0xC1
	0,					// 0xC2
	0,					// 0xC3
	0,					// 0xC4
	0,					// 0xC5
	0,					// 0xC6
	0,					// 0xC7
	0,					// 0xC8
	0,					// 0xC9
	0,					// 0xCA
	0,					// 0xCB
	0,					// 0xCC
	0,					// 0xCD
	0,					// 0xCE
	0,					// 0xCF
	0,					// 0xD0
	0,					// 0xD1
	0,					// 0xD2
	0,					// 0xD3
	0,					// 0xD4
	0,					// 0xD5
	0,					// 0xD6
	0,					// 0xD7
	0,					// 0xD8
	0,					// 0xD9
	0,					// 0xDA
	QzKey_LeftBracket,	// 0xDB VK_OEM_4		[{
	QzKey_BackSlash,	// 0xDC VK_OEM_5		\|
	QzKey_RightBracket,	// 0xDD VK_OEM_6		]}
	QzKey_Apostrophe,	// 0xDE VK_OEM_7		'"
	0,					// 0xDF
	0,					// 0xE0
	0,					// 0xE1
	0,					// 0xE2
	0,					// 0xE3
	0,					// 0xE4
	0,					// 0xE5
	0,					// 0xE6
	0,					// 0xE7
	0,					// 0xE8
	0,					// 0xE9
	0,					// 0xEA
	0,					// 0xEB
	0,					// 0xEC
	0,					// 0xED
	0,					// 0xEE
	0,					// 0xEF
	0,					// 0xF0
	0,					// 0xF1
	0,					// 0xF2
	0,					// 0xF3
	0,					// 0xF4
	0,					// 0xF5
	0,					// 0xF6
	0,					// 0xF7
	0,					// 0xF8
	0,					// 0xF9
	0,					// 0xFA
	0,					// 0xFB
	0,					// 0xFC
	0,					// 0xFD
	0,					// 0xFE
	0					// 0xFF
};


/////////////////////////////////////////////////////////////////////////////
//
//	LowLevelKeyboardProc()
//
//	Need to inject a low-level keyboard handler to trap the Windows keys and
//	print screen button.
//
LRESULT CALLBACK LowLevelKeyboardProc(int code, WPARAM wParam, LPARAM lParam)
{
	// Do not attempt to process non-keyboard operations.
	if ((code < 0) || (HC_ACTION != code)) {
		return CallNextHookEx(g_hKeyboardHook, code, wParam, lParam);
	}

	KBDLLHOOKSTRUCT* p = reinterpret_cast<KBDLLHOOKSTRUCT*>(lParam);

	// Key up and down events may need to be highjacked.
	if ((WM_KEYDOWN == wParam) || (WM_KEYUP == wParam)) {
		// If the app is in full screen mode,
		// and the app is monopolizing the keyboard (e.g., a game in FPS mode),
		// and the user wants the Windows keys disabled in this mode,
		// then we can return 1 to prevent Windows from processing the
		// key-press and displaying the Start menu.
		//
		// Note: If you wanted to be a bit more helpful, you could allow
		// independent control of left and right Windows keys (e.g., allow
		// the user to disable just the left one, due to proximity to WASD
		// keys, but leave the right one working so the user can still
		// bring up the start menu and change window focus).
		//
		if (g_FullScreen && g_IgnoreWindowsKey && g_AllowIgnoreWindowsKey) {
			if ((VK_LWIN == p->vkCode) || (VK_RWIN == p->vkCode)) {
				return 1;
			}
		}

		// Uncomment this code to disable the native Windows print-screen
		// functionality, otherwise the app will never see print-screen key
		// presses from the keyboard.
		//
		// If this is uncommented, the app becomes responsible for detecting
		// any QzKey_PrintScreen events.  Usually this would entail rendering
		// to a buffer and saving the result out to a screen shot directory.
		//
//		if (VK_SNAPSHOT == p->vkCode) {
//			return 1;
//		}
	}

	return CallNextHookEx(g_hKeyboardHook, code, wParam, lParam);
}


/////////////////////////////////////////////////////////////////////////////
//
//	constructor
//
QzKeyMapper::QzKeyMapper(void)
	:	m_pPrivate(NULL)
{
	m_pPrivate = new DevInputPrivate_t;

	DevInputPrivate_t *pPrivate = reinterpret_cast<DevInputPrivate_t*>(m_pPrivate);

	pPrivate->StickyKeysDisabled = false;
	pPrivate->StickyKeys.cbSize  = sizeof(STICKYKEYS);
	pPrivate->StickyKeys.dwFlags = 0;
	pPrivate->ToggleKeys.cbSize  = sizeof(TOGGLEKEYS);
	pPrivate->ToggleKeys.dwFlags = 0;
	pPrivate->FilterKeys.cbSize  = sizeof(FILTERKEYS);
	pPrivate->FilterKeys.dwFlags = 0;

	// Save the current sticky/toggle/filter key settings so they can be
	// restored when shutting down or alt-tabbing away.
	SystemParametersInfo(SPI_GETSTICKYKEYS, sizeof(STICKYKEYS), &(pPrivate->StickyKeys), 0);
	SystemParametersInfo(SPI_GETTOGGLEKEYS, sizeof(TOGGLEKEYS), &(pPrivate->ToggleKeys), 0);
	SystemParametersInfo(SPI_GETFILTERKEYS, sizeof(FILTERKEYS), &(pPrivate->FilterKeys), 0);

	// Zero out the key state information.
	Reset();

	// Hook in the keyboard handler so we can trap low-level nuisance keys
	// before they are processed by the system.
    g_hKeyboardHook = SetWindowsHookEx(WH_KEYBOARD_LL, LowLevelKeyboardProc, GetModuleHandle(NULL), 0);
}


/////////////////////////////////////////////////////////////////////////////
//
//	destructor
//
QzKeyMapper::~QzKeyMapper(void)
{
	// Need to remove the windows keyboard hook and make certain that the
	// sticky keys are re-enabled if they needed to be disabled.
	UnhookWindowsHookEx(g_hKeyboardHook);
	EnableStickyKeys();

	DevInputPrivate_t *pPrivate = reinterpret_cast<DevInputPrivate_t*>(m_pPrivate);

	SafeDelete(pPrivate);
}


/////////////////////////////////////////////////////////////////////////////
//
//	KeyPress()
//
//	This is used to record a key-press event.  This allows the key mapper to
//	keep track of which keys are currently pressed, and to record if any of
//	them were either pressed or released since the last call to ClearEvents().
//
void QzKeyMapper::KeyPress(U32 key, bool down)
{
	// This should never happen... at least not with an English keyboard.
	// What happens with non-Qwerty, non-English keyboards?
	if (key >= ArraySize(g_KeyMap)) {
		return;
	}

	// Translate the virtual key into something more useful.
	U32 keyMap = g_KeyMap[key];

	// Always keep the current key state up-to-date.
	if (down) {
		// Only set the "was down" flag if the key is not already down.
		// This will ignore key repeats when the key is already down due to
		// the player holding the key down for long periods of time (such as
		// the movement keys).
		if (0 == (QzKeyMask_IsDown & m_KeyState[keyMap].Flags)) {
			m_KeyState[keyMap].Flags |= QzKeyMask_IsDown | QzKeyMask_WasDown;
		}

		m_KeyState[keyMap].PressCount += 1;
	}
	else {
		m_KeyState[keyMap].Flags &= ~QzKeyMask_IsDown;
		m_KeyState[keyMap].Flags |=  QzKeyMask_WasUp;
	}

	// Append the key event to the buffer, but be mindful of overflow.
	// Overflow should not happen unless a lock-up prevented rendering
	// for a protracted period of time.
	if (m_KeyEventCount < ArraySize(m_KeyEvents)) {
		m_KeyEvents[m_KeyEventCount].ID          = U16(keyMap);
		m_KeyEvents[m_KeyEventCount].Down        = down ? TRUE : FALSE;
		m_KeyEvents[m_KeyEventCount].RepeatCount = 1;
		++m_KeyEventCount;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetKeyState()
//
//	Return the up/down/held state for the key.  This is normally used to test
//	key state for keys that will be pressed and held for periods of time,
//	such as movement keys.
//
U32 QzKeyMapper::GetKeyState(U32 key)
{
	if (key < ArraySize(m_KeyState)) {
		return m_KeyState[key].Flags;
	}

	return 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	GetEvents()
//
//	Returns the list of key presses that have been received since the last
//	update.  This will store the QzKey_... enum value in the low bits, and
//	set the high bit if it was a key-down event.
//
//	This is normally used to determine if trigger-type keys were pressed
//	since the last frame (quick keys, function keys, etc.).
//
U32 QzKeyMapper::GetEvents(U32 events[], U32 maxEvents)
{
	U32 eventCount = maxEvents;

	if (eventCount > m_KeyEventCount) {
		eventCount = m_KeyEventCount;
	}

	for (U32 i = 0; i < eventCount; ++i) {
		events[i] = m_KeyEvents[i].ID;

		if (m_KeyEvents[i].Down) {
			events[i] |= 0x80000000;
		}
	}

	return eventCount;
}


/////////////////////////////////////////////////////////////////////////////
//
//	TranslateKey()
//
//	This will be called for all key press events.  It is needed to map the
//	virtual key values to standard, cross-platform values.  It also fixes
//	the flag bits to properly indicate which control/shift/alt keys are
//	currently being held down.
//
bool QzKeyMapper::TranslateKey(U32 &flags, U32 &key)
{
	if (key >= ArraySize(g_KeyMap)) {
		return false;
	}

	flags = 0;

	if (0 != (QzKeyMask_IsDown & GetKeyState(QzKey_Alt))) {
		flags |= QzKeyMask_Alt;
	}

	if (0 != (QzKeyMask_IsDown & GetKeyState(QzKey_Control))) {
		flags |= QzKeyMask_Control;
	}

	if (0 != (QzKeyMask_IsDown & GetKeyState(QzKey_NumLock))) {
		flags |= QzKeyMask_NumLock;
	}

	if (0 != (QzKeyMask_IsDown & GetKeyState(QzKey_ScrollLock))) {
		flags |= QzKeyMask_ScrollLock;
	}

	if (0 != (QzKeyMask_IsDown & GetKeyState(QzKey_Shift))) {
		flags |= QzKeyMask_Shift;
	}

	// CapsLock is a persistent state that persists independently
	// from any key, so it is not possible to track that state by
	// watching for key strokes -- the state could change while this
	// app does not have the focus, so it never sees the key press.
	//
	// The CapsLock key is down if the low bit is set, which is
	// inconsistent with how other keys respond to GetKeyState().
	//
	if (::GetKeyState(VK_CAPITAL) & 1) {
		flags |= QzKeyMask_CapsLock;
	}

	// Translate the virtual key into our own key mapping.
	key = g_KeyMap[key];

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Reset()
//
//	Resets state info for all keys.  This should be called any time the
//	window is reactivated from being either in the background or minimized.
//
void QzKeyMapper::Reset(void)
{
	ClearEvents();

	memset(m_KeyState, 0, sizeof(m_KeyState));
}


/////////////////////////////////////////////////////////////////////////////
//
//	ClearEvents()
//
void QzKeyMapper::ClearEvents(void)
{
	m_KeyEventCount = 0;

	for (U32 i = 0; i < QzKey_ArraySize; ++i) {
		m_KeyState[i].Flags     &= ~(QzKeyMask_WasDown | QzKeyMask_WasUp);
		m_KeyState[i].PressCount = 0;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	SetFullScreen()
//
//	Set the static global full-screen flag.  This needs to be global so the
//	keyboard handler can see the flag.
//
void QzKeyMapper::SetFullScreen(bool fullScreen)
{
	g_FullScreen = fullScreen;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DisableWindowsKey()
//
//	This is used by the app to selectively disable the Windows key.  The app
//	could disable the Windows key all of the time, only when in full-screen
//	more, or only when playing (but leaving it enabled when in a game menu
//	screen).
//
void QzKeyMapper::DisableWindowsKey(void)
{
	g_IgnoreWindowsKey = true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	EnableWindowsKey()
//
void QzKeyMapper::EnableWindowsKey(void)
{
	g_IgnoreWindowsKey = false;
}


/////////////////////////////////////////////////////////////////////////////
//
//	DisableStickyKeys()
//
//	Disables the sticky-key functionality attached to Shift, Control, and
//	Alt keys.  Most players hate having these pop up a dialog when they're
//	playing a game, so we can do this to selectively disable keys while the
//	game is running.
//
void QzKeyMapper::DisableStickyKeys(void)
{
	DevInputPrivate_t *pPrivate = reinterpret_cast<DevInputPrivate_t*>(m_pPrivate);

	// Disable StickyKeys/etc shortcuts.  This is based off of the initial
	// state of the sticky keys.  If they were enabled, they will not be
	// disabled.  If previous disabled, do nothing.

	STICKYKEYS skOff = pPrivate->StickyKeys;
	if (0 == (skOff.dwFlags & SKF_STICKYKEYSON)) {
		// Disable the hotkey and the confirmation
		skOff.dwFlags &= ~SKF_HOTKEYACTIVE;
		skOff.dwFlags &= ~SKF_CONFIRMHOTKEY;

		SystemParametersInfo(SPI_SETSTICKYKEYS, sizeof(STICKYKEYS), &skOff, 0);
	}

	TOGGLEKEYS tkOff = pPrivate->ToggleKeys;
	if (0 == (tkOff.dwFlags & TKF_TOGGLEKEYSON)) {
		// Disable the hotkey and the confirmation
		tkOff.dwFlags &= ~TKF_HOTKEYACTIVE;
		tkOff.dwFlags &= ~TKF_CONFIRMHOTKEY;

		SystemParametersInfo(SPI_SETTOGGLEKEYS, sizeof(TOGGLEKEYS), &tkOff, 0);
	}

	FILTERKEYS fkOff = pPrivate->FilterKeys;
	if (0 == (fkOff.dwFlags & FKF_FILTERKEYSON)) {
		// Disable the hotkey and the confirmation
		fkOff.dwFlags &= ~FKF_HOTKEYACTIVE;
		fkOff.dwFlags &= ~FKF_CONFIRMHOTKEY;

		SystemParametersInfo(SPI_SETFILTERKEYS, sizeof(FILTERKEYS), &fkOff, 0);
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	EnableStickyKeys()
//
//	Restores the original sticky-key states that were in place when the app
//	started up.  We must do this to avoid leaving the stick-keys disabled,
//	which would cause problems for users who want the accessibility shortcuts
//	to work for other apps.
//
void QzKeyMapper::EnableStickyKeys(void)
{
	DevInputPrivate_t *pPrivate = reinterpret_cast<DevInputPrivate_t*>(m_pPrivate);

	// Restore StickyKeys/etc to original state and enable Windows key
	SystemParametersInfo(SPI_SETSTICKYKEYS, sizeof(STICKYKEYS), &(pPrivate->StickyKeys), 0);
	SystemParametersInfo(SPI_SETTOGGLEKEYS, sizeof(TOGGLEKEYS), &(pPrivate->ToggleKeys), 0);
	SystemParametersInfo(SPI_SETFILTERKEYS, sizeof(FILTERKEYS), &(pPrivate->FilterKeys), 0);
}


