/////////////////////////////////////////////////////////////////////////////
//
//	File: QzBuffer.cpp
//
//	$Header: /Projects/Qz/QzBuffer.cpp  7  2009/9/7 1:42:11p  Lee $
//
//
//	A simple utility class used to allocate and automatically free buffers
//	when going out of scope.
//
/////////////////////////////////////////////////////////////////////////////


#include "QzCommon.h"
#include "QzBuffer.h"


#ifdef USE_MALLOC_MACRO
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
//
//	Allocate()
//
//	Allocates a new buffer, freeing any previous one that may have been
//	present.  Returns the address of the new buffer, or NULL if allocation
//	failed.
//
U08* QzBuffer::Allocate(U32 size)
{
	Free();

	if (0 == size) {
		return NULL;
	}

	m_pBuffer = new U08[size];
	m_Size    = size;

	return m_pBuffer;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Copy()
//
//	Copies the contents of pData into the QzBuffer, forcing the buffer
//	to grow if necessary to contain the new data.
//
void QzBuffer::Copy(U08 *pData, U32 size)
{
	if (size > m_Size) {
		Allocate(size);
	}

	m_Size = size;
	memcpy(m_pBuffer, pData, size);
}


/////////////////////////////////////////////////////////////////////////////
//
//	Copy()
//
//	Copies the contents of pData into the QzBuffer, forcing the buffer
//	to grow if necessary to contain the new data.
//
void QzBuffer::Copy(QzBuffer &src)
{
	if (NULL != src.m_pBuffer) {
		SafeDeleteArray(m_pBuffer);

		m_Size    = src.m_Size;
		m_pBuffer = new U08[src.m_Size];
		memcpy(m_pBuffer, src.m_pBuffer, m_Size);
	}
	else {
		Free();
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Assign()
//
void QzBuffer::Assign(QzBuffer &src)
{
	SafeDeleteArray(m_pBuffer);

	m_Size    = src.m_Size;
	m_pBuffer = src.Extract();
}


/////////////////////////////////////////////////////////////////////////////
//
//	Assign()
//
//	Replace the existing buffer with a new one.  This object takes over
//	memory management responsibility for the new buffer.
//
void QzBuffer::Assign(U08 *pData, U32 byteCount)
{
	Free();
	m_pBuffer = pData;
	m_Size    = byteCount;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Extract()
//
//	Removes the managed memory from the buffer, handing it off to the
//	caller who becomes responsible for freeing it.
//
U08* QzBuffer::Extract(void)
{
	U08 *p    = m_pBuffer;
	m_pBuffer = NULL;
	m_Size    = 0;

	return p;
}


/////////////////////////////////////////////////////////////////////////////
//
//	Resize()
//
//	Checks the size of the buffer, forcing to to grow via reallocation,
//	preserving the current contents.  This will only increase the size
//	of the buffer, not decrease it.
//
void QzBuffer::Resize(U32 size)
{
	if (size > m_Size) {
		U08 *pTemp = new U08[size];

		memcpy(pTemp, m_pBuffer, m_Size);

		delete [] m_pBuffer;

		m_pBuffer = pTemp;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Truncate()
//
void QzBuffer::Truncate(U32 size)
{
	if (m_Size > size) {
		m_Size = size;
	}
}


/////////////////////////////////////////////////////////////////////////////
//
//	Free()
//
//	Explicitly frees the buffer.  This is also used from within the class
//	to automatically free the old buffer when allocating a new one or
//	when being destructed.
//
void QzBuffer::Free(void)
{
	SafeDeleteArray(m_pBuffer);

	m_Size = 0;
}


/////////////////////////////////////////////////////////////////////////////
//
//	ReadFromFile()
//
bool QzBuffer::ReadFromFile(const Utf08_t filename[])
{
	Free();

	m_pBuffer = QzReadFileToBuffer(filename, m_Size);

	if (NULL == m_pBuffer) {
		m_Size = 0;
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
//
//	WriteToFile()
//
void QzBuffer::WriteToFile(const Utf08_t filename[])
{
	FILE *pFile = QzFileOpen(filename, QzFileOpen_Write);

	if (NULL != pFile) {
		if (NULL != m_pBuffer) {
			fwrite(m_pBuffer, 1, m_Size, pFile);
		}
		fclose(pFile);
	}
}




